#include <EEPROM.h>
#include <GyverButton.h>
#include <Project154.h>
#include <TM1637mod.h>
#include <IC746.h>
#include <Wire.h>
#include <PT2257.h>

#include "MorseMap.cpp"
#include "config.h"

/* Constants */

#define Rx              2
#define Tx              3
#define Cw              4
#define Tune            5
#define Auto            6

const char* const beaconText[] PROGMEM = {
  beaconText0,
  beaconText1,
  beaconText2,
  beaconText3,
  beaconText4,
  beaconText5,
  beaconText6,
  beaconText7,
  beaconText8,
  beaconText9,
};
const uint16_t arrayEEPROM[] = {  //      Address in memory cells of EEPROM:
  500,                            // [0]  XTAL_CORRECT (4 bytes)
  504,                            // [1]  SI5351BX_CPF (1 byte)
  505,                            // [2]  calibration Vbatt (4 bytes)
  509,                            // [3]  Tone CW 300-2400 Hz (4 bytes)
  513,                            // [4]  brightness of LED indicator (1 byte)
  514,                            // [5]  soundLevelRX 0-79 (1 byte)
  515,                            // [6]  soundLevelTX 0-79 (1 byte)
  516,                            // [7]  soundLevelCW 0-79 (1 byte)
  517,                            // [8]  beeperButtons 0-1 (1 byte)
  518,                            // [9]  beeperSubmenu 0-1 (1 byte)
  0,                              // [10] Channels 0-99 F,SSB (4+1 bytes)*100
  519,                            // [11] beaconTextNumber 0-9 (1 byte)
  520,                            // [12] beaconCadence 30-900 s (2 bytes)
  522,                            // [13] beaconWPM 5-20 (1 byte)
};

/* Flags */

bool     TXmode        = false;
bool     CWmode        = false;
bool     TUning        = false;
bool     beacon        = false;
bool     calibrationV  = false;
bool     displayoff    = false;
bool     powerWattSet  = false;
bool     beeperButtons;
bool     beeperSubmenu;
uint8_t  scan          = 0;  //  [0]-no scan, [1]-=1 kHz,     [2]+=1 kHz;
uint8_t  mode          = 0;  //  [1] - freq   [2] - channel   [3] - voltage
                             //  [4] - CW     [5] - SWRmeter  [6] - Powermeter
                             //  [7] - bright [8] - XTAL-F    [9] - XTAL-C
                             // [10] - tune 

/* Variables */

char beaconTextBuffer[30];
uint32_t XTAL_CORRECT;       // XTAL frequency correction constant
uint8_t  SI5351BX_CPF;       // 1:6pf  2:8pf  3:10pf
uint8_t  si5351bx_drive[3];
uint8_t  digits[5];
uint8_t  brightness;
uint32_t CW_TONE;
uint8_t  MODE_SSB = MODE_USB;

/* Counters */

uint8_t  soundLevelRX;       // 0-79; 79 - max volume, 0 - mute.
uint8_t  soundLevelTX;       // 0-79; 79 - max volume, 0 - mute.
uint8_t  soundLevelCW;       // 0-79; 79 - max volume, 0 - mute.
int16_t  volt          = 0;  // 0-1023
uint8_t  channel       = 0;  // 0-99;
uint8_t  WPM;
uint8_t  beaconWPM;
uint8_t  beaconTextNum;
uint16_t beaconCadence;
volatile uint32_t debouncePCINT2_vect;

/* The sketch itself */

PT2257 volume;
IC746 radio;
TM1637 tm1637(PIN_CLK, PIN_DIO);
GButton bc(BTN_SELECTOR);    // HIGH_PULL/LOW_PULL, NORM_OPEN/NORM_CLOSE
GButton vu(BTN_VALUE_UP);
GButton vd(BTN_VALUE_DW);
GButton su(BTN_SOUND_UP);
GButton sd(BTN_SOUND_DW);

uint32_t getSideBand() {
  return MODE_SSB == MODE_USB ? BFO_USB : BFO_LSB;
}

uint32_t setF0(uint32_t f) {
  return MODE_SSB == MODE_USB ? F0 + f: F0 - f;
}

void setVFO(bool enable) {
  si5351bx_setfreq(CLK_VFO, (enable ? getSideBand() + F0: 0));
}

void setBFO(bool enable) {
  si5351bx_setfreq(CLK_BFO, (enable ? getSideBand(): 0));
}

void setTX0(uint32_t f) {
  static uint32_t old;

  if (f != old) old = f; else return;
  if (f != 0) f = setF0(f);
  si5351bx_setfreq(CLK_TX0, f);
}

void PTTon(void) {
  digitalWrite(PIN_RX, LOW);
  delay(100);
#if (powerPWM == 1)
  analogWrite(PIN_DRIVER, powerLevel);
#else
  digitalWrite(PIN_DRIVER, HIGH);
#endif
#ifdef PIN_TX
  digitalWrite(PIN_TX, HIGH);
#endif
  TXmode = true;
}

void PTToff(void) {
#ifdef PIN_TX
  digitalWrite(PIN_TX, LOW);
#endif
#if (powerPWM == 1)
  analogWrite(PIN_DRIVER, 0);
#else
  digitalWrite(PIN_DRIVER, LOW);
#endif
  delay(100);
  digitalWrite(PIN_RX, HIGH);
  TXmode = false;
}

bool inRange(uint8_t mod) {

  uint32_t F1 = mod == Rx ? FminRX: FminTX;
  uint32_t F2 = mod == Rx ? FmaxRX: FmaxTX;

  if (F0 < F1) return false;
  if (F0 > F2) return false;
  if (MODE_SSB == MODE_LSB && F0 < F1 + 3000) return false;
  if (MODE_SSB == MODE_USB && F0 > F2 - 3000) return false;
  return true;
}

void tx(void) {
  setVFO(0);
  setBFO(0);
  pciClear();
  PTTon();
  setSoundLevel(Auto);
}

void rx(void) {
  setTX0(0);
  PTToff();
  pciSetup(BTN_KEY);
  setSoundLevel(Auto);
  setVFO(1);
  setBFO(1);
}

bool keyPress(void) {
   return (bool)digitalRead(BTN_KEY) ? false: true;
}

// Cat Interface

void showFSoff(void) {
  digits[0] = 21;
  digits[1] = 5;
  digits[2] = 26;
  digits[3] = 26;
  digits[4] = 1;
  writeToDisplay(true); // FS.--
  morseInformer((mode >= 4 && mode <= 6 ? Cw: Rx),"ERR");
}

void catSetPtt(bool catPTT) {
  if (mode > 2) {
    showFSoff(); 
    return;
  }
  if (!inRange(Tx)) {
    showFErr();
    return;
  }
  if (catPTT) {
    tx();
    adcOFF();
  } else {
    adcON();
    rx();
  }
}

void catSetMode(uint8_t mod) {
  if (TXmode) return;
  if (mode == 1) {
         if (mod == CAT_MODE_ADD) channelCreate();
    else if (mod == CAT_MODE_DEL) channelDelete();
    else {
      MODE_SSB = mod == CAT_MODE_LSB ? MODE_LSB: MODE_USB;
      setVFO(1);
      setBFO(1);
      showSideBand();
    }
  }
}

void catSetFreq(int32_t f) {
  if (TXmode) return;
  if (mode == 1) {
    if (F0 != (uint32_t)f) {
      F0 = (uint32_t)f;
      setVFO(1);
    }
  }
}

bool catGetPtt() {
  return TXmode;
}

uint8_t catGetMode() {
  return MODE_SSB == MODE_LSB ? CAT_MODE_LSB: CAT_MODE_USB;
}

long catGetFreq() {
  return (int32_t)F0;
}

// AFP-FSK mode

void pciClear() {
  noInterrupts();
  PCICR  = 0;
  PCMSK0 = 0;
  PCMSK1 = 0;
  PCMSK2 = 0;
  interrupts();
}

void pciSetup(uint8_t pin) {
  pciClear();
  noInterrupts();
  *digitalPinToPCMSK(pin) |= bit (digitalPinToPCMSKbit(pin)); // PCMSK2 = 0b00010000; // PCMSK2 |= (1 << PCINT20);
  PCIFR  |= bit (digitalPinToPCICRbit(pin));
  PCICR  |= bit (digitalPinToPCICRbit(pin));                  // PCICR  = 0b00000100; // PCICR |= (1 << PCIE2);
  interrupts();
  delay(1);
}

void adcON(void) { // RX
  ADMUX  = 0b00000000;            // REFS1 REFS0? ADLAR - MUX3 MUX2 MUX1 MUX0
  ADCSRA = 0b10000111;            // ADEN+ ADSC ADATE ADIF ADIE ADPS2+ ADPS1+ ADPS0+    =CLK/128
  ADCSRB = 0b00000000;            // - ACME+ - - - ADTS2 ADTS1 ADTS0
  ACSR   = 0b00010000;            // ACD ACBG ACO? ACI+ ACIE ACIC ACIS1 ACIS0
  noInterrupts();
  TCCR1A = 0b00000001;            // COM1A1 COM1A0 COM1B1 COM1B0 - - PWM11 PWM10+
  TCCR1B = 0b00000011;            // ICNC1 ICES1 - - CTC1 CS12 CS11+ CS10+              =CK/64
  TCCR1C = 0b00000000;            // FOC1A FOC1B - - - - - -
  TIMSK1 = 0b00000000;            // - - ICIE1 - - OCIE1B OCIE1A TOIE1
  TIFR1  = 0b00100111;            // - - ICF1+ - - OCF1B+ OCF1A+ TOV1+
  ICR1   = 0;
  interrupts();
  pciClear();
  delay(1);
}

void adcOFF(void) { // TX
  ADMUX  = 0b00000011;            // REFS1 REFS0 ADLAR - MUX3 MUX2 MUX1 MUX0             =ADC3
  ADCSRA = 0b00000000;            // ADEN ADSC ADATE ADIF ADIE ADPS2 ADPS1 ADPS0
  ADCSRB = 0b01000000;            // - ACME+ - - - ADTS2 ADTS1 ADTS0
  ACSR   = 0b01000111;            // ACD ACBG+ ACO ACI ACIE ACIC+ ACIS1+ ACIS0+
  noInterrupts();
  TCCR1A = 0b00000000;            // COM1A1 COM1A0 COM1B1 COM1B0 - - PWM11 PWM10
  TCCR1B = 0b11000001;            // ICNC1+ ICES1+ - - CTC1 CS12 CS11 CS10+              =CK/1
  TCCR1C = 0b00000000;            // FOC1A FOC1B - - - - - -
  ICR1 = 0;
  TIMSK1 = 0b00100001;            // - - ICIE1+ - - OCIE1B OCIE1A TOIE1+
  TIFR1  = 0b00000110;            // - - ICF1 - - OCF1B+ OCF1A+ TOV1
  interrupts();
}

// PWM enable

void frequencyPWM(uint16_t f) { // https://alexgyver.ru/lessons/pwm-overclock/
  noInterrupts();
  TCCR2B = 0b00000001;
  TCCR2A = f == 31400 ? 0b00000001: 0b00000011; // D3 Timer2 31400 or 62500 Hz
  interrupts();
}

void setPowerLevel(uint8_t val) {
  static uint8_t old = 0;

  if (val == old) return;
  old = val;
  analogWrite(PIN_DRIVER, val);
}

// TM1637

void val2digits(int value) {
  if (value > 30000) {
    digits[0] = 21;
    digits[1] = 22;
    digits[2] = 24;
    digits[3] = 24;
    digits[4] = 10;     // Full
    return;
  }
  if (value < 0) {
    digits[0] = 26;
    digits[1] = 26;
    digits[2] = 26;
    digits[3] = 26;
    digits[4] = 10;     // ----
    return;
  }
  value %= 10000;
  digits[0] = value / 1000; value %= 1000;
  digits[1] = value / 100;  value %= 100;
  digits[2] = value / 10;
  digits[3] = value % 10;
}

void showEmpty(void) {
  digits[0] = 10;
  digits[1] = 10;
  digits[2] = 10;
  digits[3] = 10;
  digits[4] = 10;
}

void writeToDisplay(void) {
  const uint32_t timer = 250UL;
  static uint32_t previous;
  static uint8_t  locmem[5];
  bool nochange = true;

  if (!TXmode && displayoff) showEmpty();

  for (uint8_t i = 0; i < 4; i++) {
    if (digits[i] != locmem[i]) nochange = false;
  }
  if (nochange) return;

  if (millis() - previous > timer) {
    for (uint8_t i = 0; i < 4; i++) {
      locmem[i] = digits[i];
      tm1637.point(i == digits[4] ? true: false);
      tm1637.display(i, digits[i]);
    }
    previous = millis();
  }
}

void writeToDisplay(bool onoff) {
  bool dsp = displayoff;
  
  if (onoff) displayoff = false;
  writeToDisplay();
  displayoff = dsp;
}

void ignitePoint(bool on = true) {
  if (digits[4] > 3) return;
  tm1637.point(on);
  tm1637.display(digits[4], digits[digits[4]]);
}

void blinkingPoint(void) {
  const uint32_t timer = 250UL;
  static uint32_t previous;
  static bool on = true;
  
  if (digits[4] > 3) return;
  if (millis() - previous > timer) {
    on = !on;
    ignitePoint(on);
    previous = millis();
  }
}

// ADC

int takePoints(uint8_t analog) {
  int val = analogRead(analog);
  return val;
}

int runMiddleArifm(int val) {
  const uint8_t dimension = 128;
  static uint8_t idx = 0;
  static int valArray[dimension];
  int32_t average = 0;

  valArray[idx] = val;
  if (++idx >= dimension) idx = 0;
  for (uint8_t i = 0; i < dimension; i++) {
    average += valArray[i];
  }
  average = round(average / dimension);
  return (int)average;
}

int16_t takePointsAveraged(uint8_t analog) {
  int16_t val = takePoints(analog);
          val = runMiddleArifm(val);
  return val;
}

float polynomialFitting(int16_t val) {
  float p = 0;
  float x = val;
  for (uint8_t i = 0; i < sizeof(PowerPolinom)/sizeof(float); i++) {
    p+= PowerPolinom[i] * pow(x,i);
  }
  return p;
}

// Battery

float readCalibrationBattery(void) {
  float old;
  uint16_t addr = arrayEEPROM[2];

  EEPROM.get(addr, old);
  if (old < 5.0 || old > 10.0) {
    old = 7.0;
    EEPROM.put(addr, old);
  }
  return old;
}

void writeCalibrationBattery(void) {
  float val;
  uint16_t addr = arrayEEPROM[2];

  val = (float)takePointsAveraged(Battery) / (float)volt;
  EEPROM.put(addr, val);
}

bool batteryEnabled(void) {
  int v = (int)(takePoints(Battery) / readCalibrationBattery());
  return v < 50 ? false: true;
}

void showBatteryVoltage(void) {
  if (!calibrationV) {
    volt = (int)(takePointsAveraged(Battery) / readCalibrationBattery());
  }
  val2digits(volt);
  digits[0] = calibrationV ? 22: 11;
  digits[1] = digits[1] == 0 ? 10: digits[1];
  digits[4] = 2;
  if (volt < 50) show12Vnot(); // U.not
  writeToDisplay();            // U12.5 or u12.5
}

void show12Vnot(void) {
  digits[1] = 14;
  digits[2] = 15;
  digits[3] = 25;
  digits[4] = 0;        // ~.not
}

// Power Meter

int16_t PowerWattsMeterage(void) {
  delay(10);
  float f = polynomialFitting(takePoints(Forward));
  float r = polynomialFitting(takePoints(Reflect));
        f = round((f - r) * 100);
  return int16_t(f);
}

void PowerWattsChange(int16_t watt) {
  static int16_t mem;
  static uint8_t memPowerLevel = 0;

  if (mem != powerVariable) {
    mem = powerVariable;
    memPowerLevel = 0;
    setPowerLevel(powerLevel);
  }

  if (watt > powerVariable) {
    if (powerLevel > powerLevelMin) powerLevel--;
    else {
      powerVariable++;
      return;
    }
  }
  if (watt < powerVariable) {
    if (powerLevel < powerLevelMax) powerLevel++;
    else {
      powerVariable--;
      return;
    }
  }
  if (watt >= powerVariable && powerLevel == memPowerLevel) {
    powerWattSet = true;
    return;
  }

  memPowerLevel = powerLevel;
  setPowerLevel(powerLevel);
}

void showPowerWatts(void) {
  if (!batteryEnabled()) show12Vnot();      // P.not
  else if (keyPress() && !powerWattSet) {
    int16_t watt = PowerWattsMeterage();
    PowerWattsChange(watt); 
    val2digits(runMiddleArifm(watt));
  } else val2digits(powerVariable);
    digits[0] = 12;
    digits[4] = 1;
  writeToDisplay();                         // P2.50 
}

void showPowerPoints(void) {
  uint8_t point = 0;
  
  if (!batteryEnabled()) show12Vnot();      // P.not
  else if (!keyPress()) val2digits(powerLevel);
  else {
    float f = polynomialFitting(takePoints(Forward));
    float r = polynomialFitting(takePoints(Reflect));
          f = round((f - r) * 100);
    val2digits(runMiddleArifm(int(f)));
    point = 1;
  }
    digits[0] = 12;
    digits[1] = (digits[1] == 0 && point == 0) ? 10: digits[1];
    digits[4] = point;
  writeToDisplay();                         // P2.50 
}

void showPowerWave(uint8_t direction) {
  float f = polynomialFitting(takePointsAveraged(direction));
        f = round(f * 100);
  val2digits(int(f));
  digits[0] = direction == Forward ? 23: 16;
  digits[4] = 1;
       if (!batteryEnabled()) show12Vnot(); // d.not
  else if (!keyPress()) showPoint000();     // d.000
  writeToDisplay();                         // d2.50 r0.05
}

// SWR Meter

void showPoints(uint8_t analog) {
  val2digits(takePointsAveraged(analog));
  digits[4] = 10;
  writeToDisplay();     // 1023
}

void showSWR(void) {
  float f = polynomialFitting(takePoints(Forward));
  float r = polynomialFitting(takePoints(Reflect));
  f = sqrt(f);
  r = sqrt(r);
  f = f <= r ? r * 1.1: f;
  int swr = round(100 * ((float)(f + r) / (float)(f - r)));
      swr = constrain(swr, 0, 999);
      swr = runMiddleArifm(swr);
  val2digits(swr);
  digits[0] = 5;
  digits[4] = 1;
       if (!batteryEnabled()) show12Vnot(); // S.not
  else if (!keyPress()) showPoint000();     // S.000
  writeToDisplay();                         // S1.15
}

// Frequency

void scanFrequency(void) {
  const uint32_t timer = 3000UL;
  static uint32_t previous;

  if (millis() - previous > timer) {
    previous = millis();
    F0 -= scan == 1 ? 1000UL: 0;
    F0 += scan == 2 ? 1000UL: 0;
    if (F0 > FmaxRX) F0 = FminRX;
    if (F0 < FminRX) F0 = FmaxRX;
  }
}

void FrequencyStepNormal(void) {
  F0 = (F0 / 1000) * 1000;
}

void showFrequency(void) {
  val2digits((int)(F0 / 1000));
  digits[4] = 0;
  writeToDisplay();     // 3.579
}

void showFrequency(bool onoff) {
  bool dsp = displayoff;
  
  if (onoff) displayoff = false;
  showFrequency();
  displayoff = dsp;
}

void showFrequencyKHZ(void) {
  val2digits((int)(F0 % 10000));
  digits[4] = 0;
  writeToDisplay();     // [357]9.000
}

// Channels

uint16_t getChannelAddres(void) {
  return (uint16_t)(arrayEEPROM[10] + channel * 5);
}

uint8_t channelDetect() {
  uint8_t f;

  f = F0 % 100;
  F0 -= f;
  return f;  
}

void channelCreate() {
  uint16_t addr;
  uint8_t ch = channelDetect();

  if (!inRange(Rx) && !inRange(Tx)) {
    showFErr();
    return;
  }

  channel = ch;
  addr = getChannelAddres();
  EEPROM.put(addr,   F0);
  EEPROM.put(addr+4, MODE_SSB);
  catSetFreq((int32_t)F0);

  val2digits(int(ch));
  digits[0] = 20;
  digits[1] = 18;
  digits[4] = 1;

  writeToDisplay(true); // Ch.15
  delay(300);
  showFrequency(true);  // 3.579
  delay(300);
  showSideBand();       // LSB/USB
}

void channelDelete() {
  const uint8_t no = 255;

  channel = channelDetect();
  uint16_t addr = getChannelAddres();
  EEPROM.put(addr,   no);
  EEPROM.put(addr+1, no);
  EEPROM.put(addr+2, no);
  EEPROM.put(addr+3, no);
  EEPROM.put(addr+4, no);
  catSetFreq((int32_t)F0);

  val2digits(int(channel));
  digits[0] = 19;
  digits[1] = 18;
  digits[4] = 1;
  writeToDisplay(true); // ch.15
  delay(300);
  digits[2] = 14;
  digits[3] = 15;
  writeToDisplay(true); // ch.no
  delay(300);
}

bool channelRead() {
  uint16_t addr = getChannelAddres();
  uint32_t freq;
  uint8_t  side;

  EEPROM.get(addr,   freq);
  EEPROM.get(addr+4, side);

       if (side == CAT_MODE_LSB) MODE_SSB = CAT_MODE_LSB;
  else if (side == CAT_MODE_USB) MODE_SSB = CAT_MODE_USB;
  else return false;

  if (FminRX > freq || freq > FmaxRX) return false;

  F0 = freq;
  setVFO(1);
  setBFO(1);
  return true;
}

void scanChannels(void) {
  const uint32_t timer = 3000UL;
  static uint32_t previous;

  if (millis() - previous > timer) {
    previous = millis();
    uint8_t i = channel;
NEXT_CHANNEL:
    if (scan == 1) {
       if (channel == 0) channel = 99;
       else channel --;
    }
    if (scan == 2) {
       if (channel == 99) channel = 0;
       else channel ++;
    }
    if (!channelRead()) {
      if (i == channel) scan = 0;
      else goto NEXT_CHANNEL;
    }
  }
}

void showChannel(void) {
  val2digits(int(channel));
  digits[0] = channelRead() ? 20: 19;
  digits[1] = 18;
  digits[4] = 1;
  writeToDisplay();     // Ch.15 or ch.15 or Ch 15
}

// Brightness

uint8_t readBrightness(void) {
  uint8_t old;

  uint16_t addr = arrayEEPROM[4];
  EEPROM.get(addr, old);
  return old <= 7 ? old: 4;
}

void writeBrightness(void) {
  uint16_t addr = arrayEEPROM[4];
  if (brightness <= 7) EEPROM.put(addr, brightness);
}

void showBrightness(void) {
  uint8_t old = readBrightness();
  digits[0] = 13;
  digits[1] = 16;
  digits[2] = old == brightness ? 10: 0;
  digits[3] = brightness;
  digits[4] = 1;
  writeToDisplay();     // br. 4
}

// CW - functions from GyverPWM https://github.com/AlexGyver/GyverLibs/blob/master/GyverPWM/GyverPWM.cpp

void defaultSetupD9(void) {
  TCCR1A = 0b10100001;
  TCCR1B = 0b00000011;
}

void squareSetupD9(float frequency) {
  uint32_t top;

  TCCR1A = 0b01010000;
  TCCR1B = 0b00011001;
  TCCR1C = 0b01000000;

  top = (float)CPU_CLOCK_FREQ / frequency / 2 - 1;
  ICR1H = highByte(top);
  ICR1L = lowByte(top);
}

void toneStart(void) {
  TCCR1A &= ~(1 << COM1A1);
  bitWrite(PORTB, 1, 0);
}

void toneStop(void) {
  TCCR1A |= (1 << COM1A1);
}

uint32_t readToneCW(void) {
  uint16_t addr = arrayEEPROM[3];
  uint32_t old;

  EEPROM.get(addr, old);
  return (old >= 300 && old <= 2000) ? old: 600;
}

void writeToneCW(void) {
  uint16_t addr = arrayEEPROM[3];
  uint32_t val = (CW_TONE >= 300 && CW_TONE <= 2000) ? CW_TONE: 600;
  EEPROM.put(addr, val);
}

void showCW(void) {
  int val = (int)(round(CW_TONE / 10));
  bool v = batteryEnabled();
  val2digits(val);
  digits[0] = v ? 20: 19;
  digits[4] = v ? (MODE_SSB == MODE_USB ? 11: 17): (MODE_SSB == MODE_USB ? 22: 24);
  digits[1] = digits[1] == 0 ? digits[4]: digits[1];
  digits[4] = 1;
  writeToDisplay();     // +12V: CU.60, CL.60; 
}                       //  12V down: cu.60, cl.60; [60=CW_TONE/10; L,l=LSB; U,u=USB;]

// Morse

String morseEncode(const char *str) {
  String morseWord = "";
  while(*str != '\0') {
    for (uint8_t i = 0; i < sizeof MorseMap / sizeof *MorseMap; ++i ) {
      if (*str == MorseMap[i].letter) {
        morseWord += MorseMap[i].code;
        morseWord += " ";
        break;
      }
    }
    str++;
  }
  return morseWord += "  ";
}

void morseUnit(uint8_t n) {
  toneStart();
  if (beacon && mode == 4) setTX0(CW_TONE);
    delay(WPM * n);
  if (beacon && mode == 4) setTX0(0);
  toneStop();
  delay(WPM);
}

void morseSilence(void) {
  delay(WPM * 3);
}

void morseInfoCW(const char str[]) {
  uint32_t timer = 1000UL;
  uint32_t begin = millis();
  
  String morseWord = morseEncode(str);
  for (uint8_t i = 0; i <= morseWord.length(); i++) {
    switch(morseWord[i]) {
      case '.': morseUnit(1);   break;
      case '-': morseUnit(3);   break;
      case ' ': morseSilence(); break;
    }
  }
  if (beacon && (millis() - begin) > timer * (beaconCadence - 5)) beacon = false;
}

void morseInfoRX(const char str[]) {
  setSoundLevel(Cw);
    squareSetupD9((float)CW_TONE);
      morseInfoCW(str);
    defaultSetupD9();
  setSoundLevel(Rx);
}

void morseInformer(uint8_t mod, const char str[]) {
  if (mod == Rx) morseInfoRX(str);
  if (mod == Cw) morseInfoCW(str);
}

void morseInformer(uint8_t mod, uint8_t num) {
  strcpy_P(beaconTextBuffer, (char *)pgm_read_word(&(beaconText[num])));
  morseInformer(mod,beaconTextBuffer);
}

// Beacon

bool changeWPM(bool on) {
  WPM = 1200 / (on ? beaconWPM: WPM_INFOMER);
}

uint8_t readBeaconTextNum(void) {
  uint8_t old;

  uint16_t addr = arrayEEPROM[11];
  EEPROM.get(addr, old);
  return old <= 9 ? old: 0;
}

void writeBeaconTextNum(void) {
  uint16_t addr = arrayEEPROM[11];
  if (beaconTextNum <= 9) EEPROM.put(addr, beaconTextNum);
}

uint16_t readBeaconCadence(void) {
  uint16_t old;

  uint16_t addr = arrayEEPROM[12];
  EEPROM.get(addr, old);
  return (old >= 30 && old <= 900) ? old: 60;
}

void writeBeaconCadence(void) {
  uint16_t addr = arrayEEPROM[12];
  if (beaconCadence >=30 && beaconCadence <= 900) EEPROM.put(addr, beaconCadence);
}

uint8_t readBeaconWPM(void) {
  uint8_t old;

  uint16_t addr = arrayEEPROM[13];
  EEPROM.get(addr, old);
  return (old >= 5 && old <= 20) ? old: 13;
}

void writeBeaconWPM(void) {
  uint16_t addr = arrayEEPROM[13];
  if (beaconWPM >= 5 && beaconWPM <= 20) EEPROM.put(addr, beaconWPM);
}

bool beaconRun(bool clear = false) {
  uint32_t timer = 1000UL;
  static uint32_t previous;
  
  if (clear) { previous = 0; return false; }
  if (millis() - previous > timer * beaconCadence) {
    previous = millis();
    return true;
  }
  return false;
}

void beaconTimerClear(void) {
  beaconRun(true);
}

void showBeaconCadence(void) {
  val2digits(int(beaconCadence));
  digits[0] = 13;
  digits[1] = digits[1] == 0 ? 10: digits[1];
  digits[4] = 0;
  writeToDisplay();     // b.xxx
}

void showBeaconWPM(void) {
  val2digits(int(beaconWPM));
  digits[0] = 13;
  digits[1] = 12;
  digits[2] = digits[2] == 0 ? 10: digits[2];
  digits[4] = 1;
  writeToDisplay();     // bP.xx
}

void showBeaconOff(void) {
  digits[0] = beaconTextNum;
  digits[1] = batteryEnabled() ? 0: 15;
  digits[2] = 21;
  digits[3] = 21;
  digits[4] = 0;
  writeToDisplay();     // #.OFF
}

void showBeaconRun(void) {
  digits[0] = beaconTextNum;
  digits[1] = 16;
  digits[2] = 22;
  digits[3] = 14;
  digits[4] = 0;
  if (!batteryEnabled()) show12Vnot(); // #.not
  writeToDisplay();     // #.run
}

void showBeaconHold(void){
  digits[0] = 18;
  digits[1] = 15;
  digits[2] = 24;
  digits[3] = 23;
  digits[4] = 10;
  writeToDisplay();     // hold
}

// Beeper

bool readBeeperButtons(void) {
  uint8_t old;

  uint16_t addr = arrayEEPROM[8];
  EEPROM.get(addr, old);
  return old == 0 ? false: true;
}

void writeBeeperButtons(void) {
  uint16_t addr = arrayEEPROM[8];
  EEPROM.put(addr, (beeperButtons ? 1: 0));
}

bool readBeeperSubmenu(void) {
  uint8_t old;

  uint16_t addr = arrayEEPROM[9];
  EEPROM.get(addr, old);
  return old == 0 ? false: true;
}

void writeBeeperSubmenu(void) {
  uint16_t addr = arrayEEPROM[9];
  EEPROM.put(addr, (beeperSubmenu ? 1: 0));
}

void showBeeperButtons(void) {
  digits[0] = 13;
  digits[1] = 25;
  digits[2] = 14;
  digits[3] = beeperButtons ? 1: 0;
  digits[4] = 2;
  writeToDisplay();     // btn.0 or btn.1
}

void showBeeperSubmenu(void) {
  digits[0] = 5;
  digits[1] = 22;
  digits[2] = 13;
  digits[3] = beeperSubmenu ? 1: 0;
  digits[4] = 2;
  writeToDisplay();     // Sub.0 or Sub.1
}

// Morse Buttons

char MorseKeySwitchValue(void) {

  vu.tick(); // ↑
  vd.tick(); // ↓
  
       if (vu.isSingle()) return 'E'; //  69 •
  else if (vu.isDouble()) return 'I'; //  73 ••
  else if (vu.isTriple()) return 'S'; //  83 •••
  else if (vu.isStep(0))  return 'T'; //  84 —
  else if (vu.isStep(1))  return 'A'; //  65 •—
  else if (vu.isStep(2))  return 'U'; //  85 ••—

  else if (vd.isSingle()) return 'e'; // 101 •
  else if (vd.isDouble()) return 'i'; // 105 ••
  else if (vd.isTriple()) return 's'; // 115 •••
  else if (vd.isStep(0))  return 't'; // 116 —
  else if (vd.isStep(1))  return 'a'; //  97 •—
  else if (vd.isStep(2))  return 'u'; // 117 ••—
  
  else return '0';                    //  48
}

char MorseKeySwitchSound(void) {

  su.tick(); // ↑
  sd.tick(); // ↓
  
       if (su.isSingle()) return 'E'; //  69 •
  else if (su.isDouble()) return 'I'; //  73 ••
  else if (su.isTriple()) return 'S'; //  83 •••
  else if (su.isStep(0))  return 'T'; //  84 —
  else if (su.isStep(1))  return 'A'; //  65 •—
  else if (su.isStep(2))  return 'U'; //  85 ••—

  else if (sd.isSingle()) return 'e'; // 101 •
  else if (sd.isDouble()) return 'i'; // 105 ••
  else if (sd.isTriple()) return 's'; // 115 •••
  else if (sd.isStep(0))  return 't'; // 116 —
  else if (sd.isStep(1))  return 'a'; //  97 •—
  else if (sd.isStep(2))  return 'u'; // 117 ••—
  
  else return '0';                    //  48
}

// Sound Volume Control

uint8_t readSoundLevel(uint8_t adr) {
  uint8_t old;

  uint16_t addr = arrayEEPROM[adr];
  EEPROM.get(addr, old);
  return old > 79 ? 65: old;
}

void writeSoundLevel(uint8_t val) {
  uint16_t addr;

       if (CWmode) addr = arrayEEPROM[7];
  else if (TXmode) addr = arrayEEPROM[6];
  else             addr = arrayEEPROM[5];

  if (val >=0 && val <= 79) EEPROM.put(addr, val);
}

uint8_t getSked(void) {
  return (TXmode) ? ((CWmode || TUning) ? Cw: Tune): Rx;
}

uint8_t getSoundLevel(void) {
  switch (getSked()) {
    case   Rx: return soundLevelRX; break;
    case   Cw: return soundLevelCW; break;
    case Tune: return soundLevelTX; break;
  }
}

void saveSoundLevel(uint8_t soundLevel) {
  switch (getSked()) {
    case   Rx: soundLevelRX = soundLevel; break;
    case   Cw: soundLevelCW = soundLevel; break;
    case Tune: soundLevelTX = soundLevel; break;
  }
}

void setSoundLevel(uint8_t mod) {
  switch (mod == Auto ? getSked(): mod) {
    case   Rx: volume.Right(0); volume.Left (soundLevelRX); break;
    case   Cw: volume.Left(0);  volume.Right(soundLevelCW); break;
    case Tune: volume.Left(0);  volume.Right(soundLevelTX); break;
  }
}

void sound(uint16_t delaytime) {
  static uint32_t previous;
  const  uint32_t timer = 500UL;
  bool change = true;
  uint8_t soundLevel = getSoundLevel();
  
  switch (MorseKeySwitchSound()) {
    case 't': if (soundLevel >= 10) soundLevel -= 9;
    case 'e': if (soundLevel >=  1) soundLevel -= 1; break;
    case 'T': if (soundLevel <= 69) soundLevel += 9;
    case 'E': if (soundLevel <= 78) soundLevel += 1; break;
    case 'a': soundLevel =  0; break;
    case 'A': soundLevel = 79; break;
    case 'u': volume.Mute(true);  showSoundMute(); delay(delaytime); previous = millis(); return; break;
    case 'U': volume.Mute(false); break;
    case 'i':
    case 'I': writeSoundLevel(soundLevel); showSAFE(); delay(delaytime); previous = millis(); return; break;
    default:  change = false;
  }

  if (change) {
    saveSoundLevel(soundLevel);
    setSoundLevel(Auto);
    showSoundLevel(soundLevel);
    delay(delaytime);
    previous = millis();
  }
  
  if (delaytime == 0 && millis() - previous > timer) {
    showFSon();
    previous = millis();
  }
}

void showFSon(void) {
  digits[0] = 21;
  digits[1] = 5;
  digits[2] = 15;
  digits[3] = 14;
  digits[4] = 1;
  writeToDisplay();     // FS.on
}

void showSoundLevel(uint8_t val) {
  val2digits((int)val);
  digits[0] = digits[2] == 0 ? 10: digits[2];
  digits[1] = digits[3];
  digits[2] = 23;
  digits[3] = 13;
  digits[4] = 1;
  writeToDisplay(true); // 10.db
}
void showSoundMute(void) {
  digits[0] = 26;
  digits[1] = 26;
  digits[2] = 23;
  digits[3] = 13;
  digits[4] = 1;
  writeToDisplay(true); // --.db
}

// Calibration

void showXTALF(void) {
  val2digits((int)XTAL_CORRECT);
  digits[4] = 10;
  writeToDisplay();     // 1250
}

void calibrationXTAL(bool on) {
//  calibrationXTALtxmode(on);   // calibration XTAL in tx mode
}

void calibrationXTALtxmode(bool on) {
  static bool old;
  
  if (old != on) old = on; else return;
  if (on) {
    setVFO(0);
    setBFO(0);
    analogWrite(PIN_DRIVER, 0);
    setTX0(CW_TONE);
  } else {
    rx();
  }
}

uint32_t readXTALF(void) {
  uint32_t old;
  uint16_t addr = arrayEEPROM[0];
  EEPROM.get(addr, old);
  if (old > 9999) old = 0;
  return old;
}

void writeXTALF(void) {
  uint16_t addr = arrayEEPROM[0];
  uint32_t val = (XTAL_CORRECT >= 0 && XTAL_CORRECT <= 9999) ? XTAL_CORRECT: 1000;
  EEPROM.put(addr, val);
}

void showXTALC(void) {
  val2digits((int)(4 + 2*SI5351BX_CPF));
  digits[0] = digits[2] == 0 ? 10: digits[2];
  digits[1] = digits[3];
  digits[2] = 12;
  digits[3] = 21;
  digits[4] = 1;
  writeToDisplay();     // 10.PF
}

uint8_t readXTALC(void) {
  uint8_t old;
  uint16_t addr = arrayEEPROM[1];
  EEPROM.get(addr, old);
  if (old < 1 || old > 3) old = 2;
  return old;
}

void writeXTALC(void) {
  uint16_t addr = arrayEEPROM[1];
  uint8_t val = (SI5351BX_CPF < 1 || SI5351BX_CPF > 3) ? 2: SI5351BX_CPF;
  EEPROM.put(addr, val);
}

// Displaying values and states

void showSideBand(void) {
  digits[0] = 10;
  digits[1] = MODE_SSB == MODE_LSB ? 17: 11;
  digits[2] = 5;
  digits[3] = 13;
  digits[4] = 10;
  writeToDisplay(true); // LSB or USB
  delay(500);
}

void showFErr(void) {
  digits[0] = 21;
  digits[1] = 28;
  digits[2] = 16;
  digits[3] = 16;
  digits[4] = 0;
  writeToDisplay(true); // F.Err
  morseInformer((mode >= 4 && mode <= 6 ? Cw: Rx),"ERR");
}

void showPoint000(void) {
  digits[1] = 0;
  digits[2] = 0;
  digits[3] = 0;
  digits[4] = 1;        // ~.000
}

void showSAFE(void) {
  digits[0] = 5;
  digits[1] = 27;
  digits[2] = 21;
  digits[3] = 28;
  digits[4] = 10;
  writeToDisplay(true); // SAFE
}

// Modes Switcher

void modeOn(uint8_t mod) {
  if (mod == Tune) TUning = true;
  if (mod == Cw)   CWmode = true;
  squareSetupD9((float)CW_TONE);
  tx();
}

void modeOff(uint8_t mod) {
  if (mod == Tune) TUning = false;
  if (mod == Cw)   CWmode = false;
  defaultSetupD9();
  rx();
}

void modeSwitch(uint8_t mod, uint8_t tumbler) {
  if (tumbler == 1) modeOn(mod);
  if (tumbler == 0) modeOff(mod);
}

void manipulationCW(void) {
  if ((keyPress() && powerWattSet && mode == 6) || !keyPress()) {
    setTX0(0);
    toneStop();
  } else {
    toneStart();  
    setTX0(CW_TONE);
  }
}

// INI

void correctionXTAL(bool go = false);  // calibration XTAL in tx mode - go = true;

void setup() {

  analogReference(INTERNAL);

#ifdef PIN_TX
  pinMode(PIN_TX, OUTPUT);
#endif
  pinMode(PIN_RX, OUTPUT);
  pinMode(PIN_DRIVER, OUTPUT);
  pinMode(BTN_KEY, INPUT_PULLUP);
  pinMode(PIN_TONE, OUTPUT);

  CW_TONE = readToneCW();
  initializationSI5351();

  soundLevelRX = readSoundLevel(5);
  soundLevelTX = readSoundLevel(6);
  soundLevelCW = readSoundLevel(7);
  volume.Mute(false);                  // true - on mute  | false - off mute

  brightness = readBrightness();
  tm1637.init();
  tm1637.set(brightness);
  
  WPM = 1200 / WPM_INFOMER;
  beaconWPM = readBeaconWPM();
  beaconCadence = readBeaconCadence();
  beaconTextNum = readBeaconTextNum();
  beeperButtons = readBeeperButtons();
  beeperSubmenu = readBeeperSubmenu();

#if (powerPWM == 1)
  frequencyPWM(31400);                 // D3 frequency 31400 or 62500 Hz
  powerLevel = constrain(powerLevel, powerLevelMin, powerLevelMax);
  powerVariable = constrain(powerVariable/10, 1, 250);
#else
  powermenu = 0;
#endif

  radio.addCATPtt(catSetPtt);
  radio.addCATGetPtt(catGetPtt);
  radio.addCATFSet(catSetFreq);
  radio.addCATMSet(catSetMode);
  radio.addCATGetFreq(catGetFreq);
  radio.addCATGetMode(catGetMode);

  radio.begin(BAUDRATE, SERIAL_8N1);
  rx();

}    // setup

void loop() {

  static uint16_t PrevTimer = 0;
  bool processUI = false;

  static uint8_t  powermode = 0;   // [0] - Power, [1] - Reflect,  [2] - Forward
  static uint8_t  swrmode   = 0;   // [0] - SWR,   [1] - Reflect,  [2] - Forward
  static uint16_t shift     = 0;
  static uint16_t basicmenu = 0;
  static bool flag = false;
  uint8_t beep;

  uint16_t swith = 0;
  if (TXmode) swith += 128;
  if (TUning) swith += 256;
  if (CWmode) swith += 512;

  switch (swith) {
    case 128: if ((TCNT1 - PrevTimer) >= (CPU_CLOCK_FREQ / 1000)) {
                PrevTimer = TCNT1;
                processUI = true;
                scan = 0;
                radio.check();
                sound(0);
              }
              processAudioInput(processUI);
              return;
              break;
    case 640:
    case 384: manipulationCW(); break;
  }

  sound(500);
  radio.check();
  bc.tick();
  
       if (bc.isSingle()) swith +=  1; // •
  else if (bc.isDouble()) swith +=  2; // ••
  else if (bc.isTriple()) swith +=  4; // •••
  else if (bc.isStep(0))  swith +=  8; // —
  else if (bc.isStep(1))  swith += 16; // •—
  else if (bc.isStep(2))  swith += 32; // ••—
  else if (bc.isStep(3))  swith += 64; // •••—
  
  switch (swith + shift) { // swith <= 960; shift > swith

  // main modes: Frequency → Channels ←←

    // Frequency

    case     0: mode = 1;
                beep = 11;
                basicmenu = 0;
                showFrequency();
                switch (MorseKeySwitchValue()) {
                  case 'a': if (scan == 0 && F0 >= FminRX + 100000) F0 -= 90000;
                  case 't': if (scan == 0 && F0 >= FminRX +  10000) F0 -=  9000;
                  case 'e': if (scan == 0 && F0 >= FminRX +   1000) F0 -=  1000; else scan = 0; break;
                  case 'A': if (scan == 0 && F0 <= FmaxRX - 100000) F0 += 90000;
                  case 'T': if (scan == 0 && F0 <= FmaxRX -  10000) F0 +=  9000;
                  case 'E': if (scan == 0 && F0 <= FmaxRX -   1000) F0 +=  1000; else scan = 0; break;
                  case 'u': scan = 1; break;
                  case 'U': scan = 2; break;
                  case 's': catSetMode(CAT_MODE_LSB); scan = 0; break;
                  case 'S': catSetMode(CAT_MODE_USB); scan = 0; break;
                  default : beep = 0;
                }
                if (scan > 0) { blinkingPoint(); scanFrequency(); } else ignitePoint();
                setVFO(1);
                break;
    case     1: scan = 0; shift = 1000; break;

    // Channels

    case  1000: mode = 2;
                beep = 11;
                basicmenu = 1000;
                showChannel();
                switch (MorseKeySwitchValue()) {
                  case 't': if (scan == 0 && channel >= 10) channel -= 9;
                  case 'e': if (scan == 0 && channel >=  1) channel --;   else scan = 0; break;
                  case 'T': if (scan == 0 && channel <= 89) channel += 9;
                  case 'E': if (scan == 0 && channel <= 98) channel ++;   else scan = 0; break;
                  case 'a': channel =  0; break;
                  case 'A': channel = 99; break;
                  case 'u': scan = 1; break;
                  case 'U': scan = 2; break;
                  default : beep = 0;
                }
                if (scan > 0) { blinkingPoint(); scanChannels(); } else ignitePoint();
                break;
    case  1001: scan = 0; shift = 0; break;
  
  // [••] CW mode → beacon worker → beacon time period → beacon WPM ←←

    case     2:
    case  1002: if (inRange(Tx)) { mode = 4; shift = 0; beep = 61; modeSwitch(Cw,1); } else showFErr(); break;

    // CW mode [••]

    case   640: beep = 41;
                showCW();
                switch (MorseKeySwitchValue()) {
                  case 't': if (CW_TONE >=  400) CW_TONE -= 90;
                  case 'e': if (CW_TONE >=  310) CW_TONE -= 10; break;
                  case 'T': if (CW_TONE <= 1900) CW_TONE += 90;
                  case 'E': if (CW_TONE <= 1990) CW_TONE += 10; break;
                  case 'a': CW_TONE =  300; break;
                  case 'A': CW_TONE = 2000; break;
                  case 'i':
                  case 'I': writeToneCW(); showSAFE(); break;
                  default : beep = 0;
                }
                break;
    case   641: beep = 61; shift = 100; break;

    // beacon worker [••] → [•]

    case   740: beep = 51;
                switch (MorseKeySwitchValue()) {
                  case 'A': if (!beacon) beaconTextNum = 8;
                  case 'E': if ( beacon) beep = 0; else if (beaconTextNum < 9) beaconTextNum ++; break;
                  case 'a': if (!beacon) beaconTextNum = 1;
                  case 'e': if ( beacon) beep = 0; else if (beaconTextNum > 0) beaconTextNum --; break;
                  case 'T': beacon = true;  changeWPM(beacon); beep = 0;           break;
                  case 't': beacon = false; changeWPM(beacon); beaconTimerClear(); break;
                  case 'i':
                  case 'I': if (!beacon){ writeBeaconTextNum(); showSAFE(); } else  beep = 0; break;
                  default : beep = 0;
                }
                if (!beacon) showBeaconOff();
                else {
                  if (beaconRun()) { showBeaconRun(); morseInformer(Cw,beaconTextNum); }
                  else showBeaconHold();
                }
                break;
    case   741: beep = 61; shift = 200; beacon = false; changeWPM(beacon); beaconTimerClear(); break;

    // beacon time period [••] → [•] → [•]

    case   840: beep = 51;
                showBeaconCadence();
                switch (MorseKeySwitchValue()) {
                  case 'u': beaconCadence =   0; break;
                  case 'U': beaconCadence = 900; break;
                  case 'a': if (beaconCadence >=  30 +  100) beaconCadence -=  90;
                  case 't': if (beaconCadence >=  30 +   10) beaconCadence -=   9;
                  case 'e': if (beaconCadence >=  30 +    1) beaconCadence -=   1; break;
                  case 'A': if (beaconCadence <= 900 -  100) beaconCadence +=  90;
                  case 'T': if (beaconCadence <= 900 -   10) beaconCadence +=   9;
                  case 'E': if (beaconCadence <= 900 -    1) beaconCadence +=   1; break;
                  case 'i':
                  case 'I': writeBeaconCadence(); showSAFE(); break;
                  default : beep = 0;
                }
                break;
    case   841: beep = 61; shift = 300; break;

    // beacon WPM [••] → [•] → [•] → [•]

    case   940: beep = 51;
                showBeaconWPM();
                switch (MorseKeySwitchValue()) {
                  case 't': if (beaconWPM >=  5 + 10) beaconWPM -=   9;
                  case 'e': if (beaconWPM >=  5 +  1) beaconWPM -=   1; break;
                  case 'T': if (beaconWPM <= 20 - 10) beaconWPM +=   9;
                  case 'E': if (beaconWPM <= 20 -  1) beaconWPM +=   1; break;
                  case 'i':
                  case 'I': writeBeaconWPM(); showSAFE(); break;
                  default : beep = 0;
                }
                break;
    case   941: beep = 61; shift = 0; break;

    case   748: beaconTimerClear(); beacon = false; changeWPM(beacon);
    case   648:
    case   848:
    case   948: beep = 22; shift = basicmenu; modeSwitch(Cw,0); break;

  // [•—] Tune SWR-meter → Tune Power-meter-points → Tune Power-meter-Watts ←←

    case    16: 
    case  1016: if (inRange(Tx)) { beep = 61; shift = powermenu; modeSwitch(Tune,1); } else showFErr(); break;

    // Tune SWR-meter [•—]

    case   384: mode = 5;
                beep = 51;
                powermenu = 0;
                switch (swrmode) {
                  case  0: showSWR();           break;
                  case  1: showPoints(Reflect); break;
                  case  2: showPoints(Forward); break;
                }
                switch (MorseKeySwitchValue()) {
                  case 't':
                  case 'T': swrmode = 0; break;
                  case 'e': swrmode = 1; break;
                  case 'E': swrmode = 2; break;
                  default : beep = 0;
                }
                break;
#if (powerPWM == 1)
    case   385: beep = 61; shift = 2000; break;

    // Tune Power-meter-points [•—] → [•]

    case  2384: if (!inRange(Tx)) { showFErr(); break; }
                mode = 5;
                beep = 51;
                powermenu = 2000;
                switch (powermode) {
                  case  0: showPowerPoints();      break;
                  case  1: showPowerWave(Reflect); break;
                  case  2: showPowerWave(Forward); break;
                }
                switch (MorseKeySwitchValue()) {
                  case 'a': if (powermode == 0 && powerLevel > powerLevelMin +100) powerLevel -= 90;
                  case 't': if (powermode == 0 && powerLevel > powerLevelMin + 10) powerLevel -=  9;
                  case 'e': if (powermode == 0 && powerLevel > powerLevelMin)      powerLevel --; powermode = 0; break;
                  case 'A': if (powermode == 0 && powerLevel < powerLevelMax -100) powerLevel += 90;
                  case 'T': if (powermode == 0 && powerLevel < powerLevelMax - 10) powerLevel +=  9;
                  case 'E': if (powermode == 0 && powerLevel < powerLevelMax)      powerLevel ++; powermode = 0; break;
                  case 's': powermode = 1; break;
                  case 'S': powermode = 2; break;
                  default : beep = 0;
                }
                setPowerLevel(powerLevel);
                break;
    case  2385: beep = 61; shift = 2100; break;

    // Tune Power-meter-Watts [•—] → [•] → [•]

    case  2484: if (!inRange(Tx)) { showFErr(); break; }
                mode = 6;
                beep = 51;
                powermenu = 2100;
                showPowerWatts();
                switch (MorseKeySwitchValue()) {
                  case 'a': if (powermode == 0 && powerVariable > 100) powerVariable -= 90;
                  case 't': if (powermode == 0 && powerVariable >  10) powerVariable -=  9;
                  case 'e': if (powermode == 0 && powerVariable >   0) powerVariable --; powermode = 0; break;
                  case 'A': if (powermode == 0 && powerVariable < 200) powerVariable += 90;
                  case 'T': if (powermode == 0 && powerVariable < 290) powerVariable +=  9;
                  case 'E': if (powermode == 0 && powerVariable < 300) powerVariable ++; powermode = 0; break;
                  default : beep = 0;
                }
                if (beep == 51) powerWattSet = false;
                break;
    case  2485: beep = 61; shift = 0; break;

    case  2392:
    case  2492:
#else
    case   385:
#endif
    case   392: beep = 22; shift = basicmenu; modeSwitch(Tune,0); break;

  // [••—] Voltmeter → Brightness → Buttons Beeper → Submenu Beeper ←←

    case    32:
    case  1032: mode = 7; beep = 11; shift = 3000; flag = displayoff; displayoff = false; break;

    // Voltmeter [••—]

    case  3000: beep = 11;
                showBatteryVoltage();
                switch (MorseKeySwitchValue()) {
                  case 't': if (volt >= 110) volt -= 9;
                  case 'e': if (volt >= 101) volt --;  calibrationV = true;  break;
                  case 'T': if (volt <= 150) volt += 9;
                  case 'E': if (volt <= 159) volt ++;  calibrationV = true;  break;
                  case 'i':
                  case 'I': writeCalibrationBattery(); calibrationV = false; showSAFE(); break;
                  default : beep = 0;
                }
                break;
    case  3001: beep = 21; shift = 4000; break;

    // Brightness [••—] → [•]

    case  4000: beep = 11;
                showBrightness();
                switch (MorseKeySwitchValue()) {
                  case 'e':
                  case 't': if (brightness > 0) brightness--; tm1637.set(brightness); break;
                  case 'E': 
                  case 'T': if (brightness < 7) brightness++; tm1637.set(brightness); break;
                  case 'i':
                  case 'I': writeBrightness(); showSAFE(); break;
                  default:  beep = 0;
                }
                break;
    case  4001: beep = 21; shift = 5000; break;

    // Buttons Beeper [••—] → [•] → [•]

    case  5000: beep = 11;
                showBeeperButtons();
                switch (MorseKeySwitchValue()) {
                  case 'E':
                  case 'T': beeperButtons = true;  break;
                  case 'e':
                  case 't': beeperButtons = false; break;
                  case 'i':
                  case 'I': writeBeeperButtons(); showSAFE(); break;
                  default : beep = 0;
                }
                break;
    case  5001: beep = 21; shift = 6000; break;

    // Submenu Beeper [••—] → [•] → [•] → [•]

    case  6000: beep = 11;
                showBeeperSubmenu();
                switch (MorseKeySwitchValue()) {
                  case 'E':
                  case 'T': beeperSubmenu = true;  break;
                  case 'e':
                  case 't': beeperSubmenu = false; break;
                  case 'i':
                  case 'I': writeBeeperSubmenu(); showSAFE(); break;
                  default : beep = 0;
                }
                break;
    case  6001: beep = 21; shift = 3000; break;

    case  3008:
    case  4008:
    case  5008:
    case  6008: beep = 22; shift = basicmenu; displayoff = flag; break;

  // [•••—] XTAL frequency calibration → XTAL capacity calibration ←←
  
    case    64:
    case  1064: mode = 8; beep = 11; shift = 7000; flag = displayoff; displayoff = false; break;

    // XTAL frequency calibration [•••—]

    case  7000: beep = 11;
                calibrationXTAL(true);
                switch (MorseKeySwitchValue()) {
                  case 'u': if (XTAL_CORRECT >=  500) XTAL_CORRECT -= 400;
                  case 'a': if (XTAL_CORRECT >=  100) XTAL_CORRECT -=  90; 
                  case 't': if (XTAL_CORRECT >=   10) XTAL_CORRECT -=   9;
                  case 'e': if (XTAL_CORRECT >=    1) XTAL_CORRECT -=   1; correctionXTAL(); break;
                  case 'U': if (XTAL_CORRECT <= 9499) XTAL_CORRECT += 400;
                  case 'A': if (XTAL_CORRECT <= 9899) XTAL_CORRECT +=  90;
                  case 'T': if (XTAL_CORRECT <= 9989) XTAL_CORRECT +=   9;
                  case 'E': if (XTAL_CORRECT <= 9998) XTAL_CORRECT +=   1; correctionXTAL(); break;
                  case 'i':
                  case 'I': writeXTALF(); showSAFE(); break;
                  default:  beep = 0;
                }
                showXTALF();
                break;
    case  7001: beep = 21; shift = 8000; break;

    // XTAL capacity calibration [•••—] → [•]

    case  8000: beep = 11;
                mode = 9;
                calibrationXTAL(true);
                switch (MorseKeySwitchValue()) {
                  case 'e':
                  case 't': if (SI5351BX_CPF > 1) SI5351BX_CPF--; correctionXTAL(); break;
                  case 'E':
                  case 'T': if (SI5351BX_CPF < 3) SI5351BX_CPF++; correctionXTAL(); break;
                  case 'i':
                  case 'I': writeXTALC(); showSAFE(); break;
                  default:  beep = 0;
                }
                showXTALC();
                break;
    case  8001: beep = 21; shift = 7000; break;

    case  7008:
    case  8008: beep = 22; shift = basicmenu; displayoff = flag; calibrationXTAL(false); break;

  // Frequency tuning [•••]

    case     4: 
    case  1004: mode = 10; beep = 11; shift = 9000; flag = displayoff; displayoff = false; break;
    
    case  9000: beep = 11;
                showFrequencyKHZ();
                switch (MorseKeySwitchValue()) {
                  case 'u': if (F0 >= FminRX + 500) F0 -= 400;
                  case 'a': if (F0 >= FminRX + 100) F0 -=  90;
                  case 't': if (F0 >= FminRX + 10)  F0 -=   9;
                  case 'e': if (F0 >= FminRX +  1)  F0 -=   1; break;
                  case 'U': if (F0 <= FmaxRX - 500) F0 += 400;
                  case 'A': if (F0 <= FmaxRX - 100) F0 +=  90;
                  case 'T': if (F0 <= FmaxRX - 10)  F0 +=   9;
                  case 'E': if (F0 <= FmaxRX -  1)  F0 +=   1; break;
                  case 's': catSetMode(CAT_MODE_LSB); break;
                  case 'S': catSetMode(CAT_MODE_USB); break;
                  default : beep = 0;
                }
                setVFO(1);
                break;

    case  9001:
    case  9008: beep = 22; shift = basicmenu; FrequencyStepNormal(); displayoff = flag; break;
  }

  if (mode > 2) scan = 0;
  if (beeperButtons) {
    if (beep == 11) morseInformer(Rx,"E");
    if (beep == 51) morseInformer(Cw,"E");
  }
  if (beeperSubmenu) {
    if (beep == 21) morseInformer(Rx,"E");
    if (beep == 22) morseInformer(Rx,"I");
    if (beep == 61) morseInformer(Cw,"E");
  }
  if (beep == 41) { squareSetupD9((float)CW_TONE); morseInformer(Cw,"E"); }

}   // loop

ISR(PCINT2_vect) {
  if (millis() - debouncePCINT2_vect >= 20 && keyPress()) {
    debouncePCINT2_vect = millis();
    displayoff = !displayoff;
  }
}

// ****************************  SI5315 routines - tks Jerry Gaffke, KE7ER   ****************************

// An minimalist standalone set of Si5351 routines.
// VCOA is fixed at 875mhz, VCOB not used.
// The output msynth dividers are used to generate 3 independent clocks
// with 1hz resolution to any frequency between 4khz and 109mhz.

// Usage:
// Call si5351bx_init() once at startup with no args;
// Call si5351bx_setfreq(clknum, freq) each time one of the
// three output CLK pins is to be updated to a new frequency.
// A freq of 0 serves to shut down that output clock.

// The global variable si5351bx_vcoa starts out equal to the nominal VCOA
// frequency of 25mhz*35 = 875000000 Hz.  To correct for 25mhz crystal errors,
// the user can adjust this value.  The vco frequency will not change but
// the number used for the (a+b/c) output msynth calculations is affected.
// Example:  We call for a 5mhz signal, but it measures to be 5.001mhz.
// So the actual vcoa frequency is 875mhz*5.001/5.000 = 875175000 Hz,
// To correct for this error:     si5351bx_vcoa=875175000;

// Most users will never need to generate clocks below 500khz.
// But it is possible to do so by loading a value between 0 and 7 into
// the global variable si5351bx_rdiv, be sure to return it to a value of 0
// before setting some other CLK output pin.  The affected clock will be
// divided down by a power of two defined by  2**si5351_rdiv
// A value of zero gives a divide factor of 1, a value of 7 divides by 128.
// This lightweight method is a reasonable compromise for a seldom used feature.

// Copyed from https://raw.githubusercontent.com/afarhan/ubitxv6/master/ubitx_si5351.cpp

#define BB0(x) ((uint8_t)x)                    // Bust int32 into Bytes
#define BB1(x) ((uint8_t)(x>>8))
#define BB2(x) ((uint8_t)(x>>16))

#define SI5351BX_ADDR   0x60                   // I2C address of Si5351   (typical)
#define SI5351BX_MSA    35                     // VCOA is at 25mhz*35 = 875mhz

uint32_t SI5351BX_XTAL = XTAL_SI5351 + XTAL_CORRECT;           // SI5351a crystal frequency in Hz
uint32_t si5351bx_vcoa = (SI5351BX_XTAL*SI5351BX_MSA);         // 25mhzXtal calibrate
uint8_t si5351bx_rdiv  = 0;                    // 0-7, CLK pin sees fout/(2**rdiv)
uint8_t si5351bx_clken = 0xFF;                 // Private, all CLK output drivers off

void i2cWrite(uint8_t reg, uint8_t val) {      // write reg via i2c
  Wire.beginTransmission(SI5351BX_ADDR);
  Wire.write(reg);
  Wire.write(val);
  Wire.endTransmission();
}

void i2cWriten(uint8_t reg, uint8_t *vals, uint8_t vcnt) {  // write array
  Wire.beginTransmission(SI5351BX_ADDR);
  Wire.write(reg);
  while (vcnt--) Wire.write(*vals++);
  Wire.endTransmission();
}

void si5351bx_init() {                         // Call once at power-up, start PLLA
  uint8_t   reg;
  uint32_t  msxp1;
  
  Wire.begin();
  i2cWrite(149, 0);                            // SpreadSpectrum off
  i2cWrite(3, si5351bx_clken);                 // Disable all CLK output drivers
  i2cWrite(183, ((SI5351BX_CPF << 6) | 0x12)); // Set 25mhz crystal load capacitance (tks Daniel KB3MUN)
  msxp1 = 128 * SI5351BX_MSA - 512;            // and msxp2=0, msxp3=1, not fractional
  uint8_t  vals[8] = {0, 1, BB2(msxp1), BB1(msxp1), BB0(msxp1), 0, 0, 0};
  i2cWriten(26, vals, 8);                      // Write to 8 PLLA msynth regs
  i2cWrite(177, 0x20);                         // Reset PLLA  (0x80 resets PLLB)
  // for (reg=16; reg<=23; reg++) i2cWrite(reg, 0x80);    // Powerdown CLK's
  // i2cWrite(187, 0);                         // No fannout of clkin, xtal, ms0, ms4
                                               // initializing the ppl2 as well
  i2cWriten(34, vals, 8);                      // Write to 8 PLLA msynth regs
  i2cWrite(177, 0xa0);                         // Reset PLLA  & PPLB (0x80 resets PLLB)
}

void si5351bx_setfreq(uint8_t clknum, uint32_t fout) {  // Set a CLK to fout Hz
  uint32_t  msa;
  uint32_t  msb;
  uint32_t  msc;
  uint32_t  msxp1;
  uint32_t  msxp2;
  uint32_t  msxp3p2top;

  if ((fout < 500000) || (fout > 109000000)) { // If clock freq out of range
    si5351bx_clken |= 1 << clknum;             //  shut down the clock
  } else {
    msa = si5351bx_vcoa / fout;                // Integer part of vco/fout
    msb = si5351bx_vcoa % fout;                // Fractional part of vco/fout
    msc = fout;                                // Divide by 2 till fits in reg
    while (msc & 0xfff00000) {
      msb = msb >> 1;
      msc = msc >> 1;
    }
    msxp1 = (128 * msa + 128 * msb / msc - 512) | (((uint32_t)si5351bx_rdiv) << 20);
    msxp2 =  128 * msb - 128 * msb / msc * msc; // msxp3 == msc;
    msxp3p2top = (((msc & 0x0F0000) << 4) | msxp2);     // 2 top nibbles
    uint8_t vals[8] = { BB1(msc), BB0(msc), BB2(msxp1), BB1(msxp1),
                        BB0(msxp1), BB2(msxp3p2top), BB1(msxp2), BB0(msxp2)
                      };
    i2cWriten(42 + (clknum * 8), vals, 8);     // Write to 8 msynth regs
    i2cWrite(16 + clknum, 0x0C | si5351bx_drive[clknum]); // use local msynth
    si5351bx_clken &= ~(1 << clknum);          // Clear bit to enable clock
  }
  i2cWrite(3, si5351bx_clken);                 // Enable/disable clock
}

// ********************************** End of Jerry's si5315bx routines **********************************

void correctionXTAL(bool txmode) {
  SI5351BX_XTAL = XTAL_SI5351 + XTAL_CORRECT;
  si5351bx_vcoa = (SI5351BX_XTAL*SI5351BX_MSA);
  si5351bx_init();
  if (txmode) si5351bx_setfreq(CLK_TX0, setF0(CW_TONE));
  else {
     setVFO(1);
     setBFO(1);
  }
}

void initializationSI5351(void) {
  XTAL_CORRECT = readXTALF();
  SI5351BX_CPF = readXTALC();
  si5351bx_drive[CLK_VFO] = POWER_VFO;
  si5351bx_drive[CLK_BFO] = POWER_BFO;
  si5351bx_drive[CLK_TX0] = POWER_TX0;
  correctionXTAL(false);
}

// *****************  Audio Wave Length Measurement routines - Kazuhisa Terasaki AG6NS  *****************
// Copyed from https://qrpguys.com/wp-content/uploads/2021/06/fsk_multi_vfo_063021.zip

#define PLL_CALCULATION_PRECISION       4
#define MIN_INPUT_AUDIO_FREQ            285                    // minimum input audio frequency limit is  300Hz-5%
#define MAX_INPUT_AUDIO_FREQ            2835                   // maximum input audio frequency limit is 2700Hz+5%
#define UPDATE_VFO_PERIOD               (CPU_CLOCK_FREQ / 250) // update FSK frequency 250 times/sec. (every 4ms)
#define NO_SIGNAL_PERIOD_THRESHOLD      (CPU_CLOCK_FREQ / 20)  // no signal detection threshold (/20 = 50ms)
#define MIN_SAMPLE_COUNT_FOR_AVERAGING  2                      // minimum sample counts for averaging filter (2)

volatile uint8_t  gMeasuredFullWaveCount = 0;
volatile uint16_t gTimer1OverflowCounter = 0;
volatile uint32_t gCurrentTimer1InputCaptureValue = 0;
volatile uint32_t gUpperHalfLenSum = 0;
volatile uint32_t gLowerHalfLenSum = 0;

inline void resetMeasuredValues(void) {        // reset values
  noInterrupts();
  gMeasuredFullWaveCount = 0;
  gUpperHalfLenSum       = 0;
  gLowerHalfLenSum       = 0;
  interrupts();
}

inline void readAndResetMeasuredValues(
  uint32_t *currentInputCaptureValue,
  uint8_t  *fullWaveCount,
  uint32_t *upperHalfWaveLenSum,
  uint32_t *lowerHalfWaveLenSum) {

  noInterrupts();
  *currentInputCaptureValue = gCurrentTimer1InputCaptureValue;
  *fullWaveCount         = gMeasuredFullWaveCount;
  *upperHalfWaveLenSum   = gUpperHalfLenSum;
  *lowerHalfWaveLenSum   = gLowerHalfLenSum;

  gMeasuredFullWaveCount = 0;
  gUpperHalfLenSum       = 0;
  gLowerHalfLenSum       = 0;
  interrupts();
}

inline uint32_t readCurrentTimer1Value(void) {

  noInterrupts();
  uint16_t counterValue = TCNT1;
  uint32_t currentTimer1Value = ((uint32_t)gTimer1OverflowCounter << 16) | counterValue;

  if ((TIFR1 & (1 << TOV1)) && (counterValue & 0x8000) == 0x0000) {
    // timer1 overflow happened and hasn't handled it yet
    currentTimer1Value += 0x10000;
  }
  interrupts();
  return currentTimer1Value;
}

void processAudioInput(bool checkNoSignal) {
  static bool sIsTransmitting = false;

  // read the length of the last measured audio wave
  uint32_t currentInputCaptureValue;
  uint8_t  inputCaptureEvents;
  uint32_t upperWaveLenSum;
  uint32_t lowerWaveLenSum;
  readAndResetMeasuredValues(&currentInputCaptureValue, &inputCaptureEvents, &upperWaveLenSum, &lowerWaveLenSum);

  static uint32_t sLastVFOUpdatedInputCaptureValue  = 0;
  static uint16_t sCapturedWaveCount                = 0;
  static uint32_t sUpperWaveLenTotal                = 0;
  static uint32_t sLowerWaveLenTotal                = 0;
  static uint32_t sLastValidSignalInputCaptureValue = 0;

  if (inputCaptureEvents > 0) {
    sCapturedWaveCount += inputCaptureEvents;
    sUpperWaveLenTotal += upperWaveLenSum;
    sLowerWaveLenTotal += lowerWaveLenSum;

    if (sLastVFOUpdatedInputCaptureValue == 0) {
      sLastVFOUpdatedInputCaptureValue = currentInputCaptureValue;
    }

    uint32_t totalWaveLength = currentInputCaptureValue - sLastVFOUpdatedInputCaptureValue;
    if (totalWaveLength >= UPDATE_VFO_PERIOD && sCapturedWaveCount >= MIN_SAMPLE_COUNT_FOR_AVERAGING) {

      // measured audio wave length
      uint32_t averageWaveLength =
        ((sUpperWaveLenTotal << PLL_CALCULATION_PRECISION) + (sCapturedWaveCount / 2)) / sCapturedWaveCount +
        ((sLowerWaveLenTotal << PLL_CALCULATION_PRECISION) + (sCapturedWaveCount / 2)) / sCapturedWaveCount;
      // measured audio frequency
      uint32_t audioFreq = (CPU_CLOCK_FREQ << (PLL_CALCULATION_PRECISION * 2)) / averageWaveLength;
      // frequency is in 28.4 fixed point number, 0.0625Hz resolution

      if (((uint32_t)MIN_INPUT_AUDIO_FREQ << PLL_CALCULATION_PRECISION) <= audioFreq
          && audioFreq <= ((uint32_t)MAX_INPUT_AUDIO_FREQ << PLL_CALCULATION_PRECISION)
          && sLowerWaveLenTotal < sUpperWaveLenTotal
          && sUpperWaveLenTotal < (sLowerWaveLenTotal << 1)) {
        // sLowerWaveLenTotal < sUpperWaveLenTotal < sLowerWaveLenTotal * 2

        // found audio signal
        sLastValidSignalInputCaptureValue = currentInputCaptureValue;

        if (sIsTransmitting) {
          setTX0((uint32_t)(audioFreq >> PLL_CALCULATION_PRECISION));
        }

        // set this flag at here so we can ignore the first detected frequency which might include some error
        sIsTransmitting = true;
      }

      sLastVFOUpdatedInputCaptureValue = currentInputCaptureValue;
      sCapturedWaveCount = 0;
      sUpperWaveLenTotal = 0;
      sLowerWaveLenTotal = 0;
    }
  }

  if (checkNoSignal && sIsTransmitting) {
    uint32_t currentTimer1Value = readCurrentTimer1Value();
    uint32_t noSignalPeriod = currentTimer1Value - sLastValidSignalInputCaptureValue;
    if (noSignalPeriod > NO_SIGNAL_PERIOD_THRESHOLD) {

      // detected no signal period
      sLastVFOUpdatedInputCaptureValue = 0;
      sCapturedWaveCount = 0;
      sUpperWaveLenTotal = 0;
      sLowerWaveLenTotal = 0;

      resetMeasuredValues();
      setTX0(0);
      sIsTransmitting = false;
    }
  }
}

inline uint32_t readTimer1InputCaptureValue(void) {
  uint16_t counterValue = ICR1;
  uint32_t currentTimer1Value = ((uint32_t)gTimer1OverflowCounter << 16) | counterValue;

  if ((TIFR1 & (1 << TOV1)) && (counterValue & 0x8000) == 0x0000) {
    currentTimer1Value += 0x10000;             // timer1 overflow happened and hasn't handled it yet
  }
  return currentTimer1Value;
}

ISR(TIMER1_CAPT_vect) {                        // ISR priority 11
  static uint32_t sPrevInputCaptureValue;
  static uint32_t sUpperWaveLen;

  uint32_t currentInputCaptureValue = readTimer1InputCaptureValue();
  uint32_t halfWaveLen = currentInputCaptureValue - sPrevInputCaptureValue;
  uint8_t  currTCCR1B = TCCR1B;

  if (currTCCR1B & (1 << ICES1)) {             // detected Falling Audio Signal Edge (Rising Input Capture Edge)
    static uint32_t sAveUpperHalfWaveLen = 0;
    sAveUpperHalfWaveLen = (sAveUpperHalfWaveLen + sAveUpperHalfWaveLen + sAveUpperHalfWaveLen + halfWaveLen) >> 2;  // (sAveUpperHalfWaveLen * 3 + halfWaveLen) / 4;
    if (halfWaveLen < ((sAveUpperHalfWaveLen >> 2) + (sAveUpperHalfWaveLen >> 4))) {
      // (sAveUpperHalfWaveLen * 0.3125)
      // ignore ripple
      return;
    }
    sUpperWaveLen = halfWaveLen;
  } else { // detected Rising Audio Signal Edge (Falling Input Capture Edge)
    static uint32_t sAveLowerHalfWaveLen = 0;
    sAveLowerHalfWaveLen = (sAveLowerHalfWaveLen + sAveLowerHalfWaveLen + sAveLowerHalfWaveLen + halfWaveLen) >> 2;  // (sAveLowerHalfWaveLen * 3 + halfWaveLen) / 4;
    if (halfWaveLen < ((sAveLowerHalfWaveLen >> 2) + (sAveLowerHalfWaveLen >> 4))) {
      // (sAveLowerHalfWaveLen * 0.3125)
      // ignore ripple
      return;
    }

    gUpperHalfLenSum += sUpperWaveLen;
    gLowerHalfLenSum += halfWaveLen;
    gCurrentTimer1InputCaptureValue = currentInputCaptureValue;
    gMeasuredFullWaveCount++;
  }
  sPrevInputCaptureValue = currentInputCaptureValue;

  TCCR1B = currTCCR1B ^ (1 << ICES1);          // flip edge selection
  TIFR1 = (1 << ICF1);                         // clear Input Capture interrupt Flag 
                                               // (flipping the edge selection causes the unwanted interrupt)
}

ISR(TIMER1_OVF_vect) {                         // ISR priority 14
  gTimer1OverflowCounter++;
}

// ********************************** End of Kazuhisa Terasaki routines *********************************
