#include <Wire.h>

  /*  Arduino IDE 1.8.9
        Скетч использует 29832 байт (97%) памяти устройства. Всего доступно 30720 байт.
        Глобальные переменные используют 894 байт (43%) динамической памяти.
      Arduino IDE 1.6.13
        Скетч использует 30466 байт (99%) памяти устройства.
        Глобальные переменные используют 910 байт (44%) динамической памяти.
  */

   /* Constants */

  const  uint8_t  OFF        = 0;
  const  uint8_t  ON         = 1;
  const  uint8_t  Rx         = 2;
  const  uint8_t  Tx         = 3;
  const  uint8_t  Cw         = 4;
  const  uint8_t  Auto       = 5;
  const  uint8_t  Go         = 6;
  const  uint16_t GO         = 6;

  struct address {
         uint8_t  frequency;              // uint32_t memory.frequency
         uint8_t  sideband;               // uint8_t  memory.sideband
         uint8_t  brightness;             // uint8_t  memory.brightness
         uint8_t  supplyVoltage;          // uint32_t memory.supplyVoltage
         uint8_t  XTALcorrector;          // uint16_t memory.XTALcorrector
         uint8_t  XTALcapacitor;          // uint8_t  memory.XTALcapacitor
         uint8_t  soundLevelRX;           // uint8_t  memory.soundLevelRX
         uint8_t  soundLevelTX;           // uint8_t  memory.soundLevelTX
         uint8_t  soundLevelCW;           // uint8_t  memory.soundLevelCW
         uint8_t  power;                  // uint8_t  memory.power
         uint8_t  channel;                // uint8_t  memory.channel
  };

   /* Variables */

         uint8_t  brightness;             // 0-7
         uint16_t XTALcorrector;          // UI      => 0-5000 XTAL frequency correction constant
         uint8_t  XTALcapacitor;          // 1-3     => 1:6pf  2:8pf  3:10pf
         uint8_t  soundLevelRX;           // 0-79;   => 79 - max volume, 0 - mute.
         uint8_t  soundLevelTX;           // 0-79;   => 79 - max volume, 0 - mute.
         uint8_t  soundLevelCW;           // 0-79;   => 79 - max volume, 0 - mute.
         uint16_t power_mWt;              // 0-power_mWt_Max

#include "config.h"

volatile uint32_t debouncePCINT2_vect;
         uint32_t frequency;
         uint8_t  sideband;
          int16_t rit           = 0;      // +- 999 Hz;
         uint8_t  channel       = 1;      // 01-50;
         uint8_t  digits[5];
         uint8_t  menu[8]       = {1,1,1,1,1,1,1,1};
         uint8_t  soundLevel;
         uint8_t  XTAL_CAPACITOR;

   /* Flags */

         bool     tuningEnd     = false;
         bool     click;
         uint8_t  Status        = 0b0000000;  // Байт флагов режимов.



#include <adc.h>
adc adc;

void pciClear() {
  noInterrupts();
  PCICR  = 0;
  PCMSK0 = 0;
  PCMSK1 = 0;
  PCMSK2 = 0;
  interrupts();
}

void pciSetup(uint8_t pin) {
  pciClear();
  noInterrupts();
  *digitalPinToPCMSK(pin) |= bit (digitalPinToPCMSKbit(pin)); // PCMSK2 = 0b00010000; // PCMSK2 |= (1 << PCINT20);
  PCIFR  |= bit (digitalPinToPCICRbit(pin));
  PCICR  |= bit (digitalPinToPCICRbit(pin));                  // PCICR  = 0b00000100; // PCICR |= (1 << PCIE2);
  interrupts();
  delay(1);
}

   /* Байт состояния Status:
      0 бит - Led 
      1 бит - Tune
      2 бит - TX
      3 бит - CW
      4 бит - 0 бит swrmode = [0] SWR     [1] Reflect [2] Forward;
      5 бит - 1 бит swrmode
      6 бит - 0 бит scaner  = [0]-no scan [1]-=1 kHz  [2]+=1 kHz;
      7 бит - 1 бит scaner
   */

enum status_flags: uint8_t {
   Led,
   Tune,
   TX,
   CW,
};

void status(uint8_t bit, uint8_t onoff) {
  switch(onoff) {
    case  ON: bitSet(Status, bit); break;
    case OFF: bitClear(Status, bit); break;
  }
}

bool status(uint8_t bit) {
  return bit_is_set(Status, bit) ? true: false;
}

   /* Генерация кода состояния по битовой маске.
      Должно работать быстрее, чем кучка IF-ов.
      Потребно для цикла AFP-FSK.
   */

uint8_t statut(uint8_t mask) {              // Status (Scan1-Scan0-swrmode1-swrmode2-CW-TX-Tune-Led)
  return Status & mask;
}

  const  uint8_t  DOWN    = 1;
  const  uint8_t  UP      = 2;
  const  uint8_t  Reflect = 1;
  const  uint8_t  Forward = 2;

void setSWR(uint8_t val) {
  bitWrite(Status, 4, bitRead(val, 0));
  bitWrite(Status, 5, bitRead(val, 1));
}

uint8_t getSWR(void) {
  uint8_t swrmode;

  swrmode = Status >> 4;
  swrmode &= 0b00000011;
  return swrmode;
}

void setScan(uint8_t val) {
  bitWrite(Status, 6, bitRead(val, 0));
  bitWrite(Status, 7, bitRead(val, 1));
}

uint8_t getScan(void) {
  uint8_t scaner;

  scaner = Status >> 6;
  scaner &= 0b00000011;
  return scaner;
}



#include <IC746.h>
IC746 uart_0;

   /* CAT интерфейс для ICOM IC-746
      Библиотека модифицирована
      
      interface(Go)                    - инициализация интерфейса при старте.
      interface(Check)                 - взаимодействие с компьютером.
      
      https://github.com/kk4das/IC746CAT
   */

const uint8_t Check = 14;

void interface(const uint8_t key) {
  switch(key) {
    case Go:
      uart_0.addCATPtt(catSetPtt);
      uart_0.addCATGetPtt(catGetPtt);
      uart_0.addCATFSet(catSetFreq);
      uart_0.addCATGetFreq(catGetFreq);
      uart_0.addCATMSet(catSetMode);
      uart_0.addCATGetMode(catGetMode);
      uart_0.begin(BAUDRATE, SERIAL_MODE);
    break;
    case Check:
      uart_0.check();
    break;
  }
}

#include <si5351.h>
Si5351 geterodin;

   /* Интерфейс для модуля SI5351A
      device   - VFO/TX0/BFO
      sideband - CAT_MODE_LSB/CAT_MODE_USB

      quartz(Go,     uint32_t f)       - инициализация модуля с калибровочным коэффициентом.
      quartz(device, uint32_t f)       - изменение частоты на выходе VFO/TX0/BFO
      quartz(device, ON/OFF)           - включение/выключение выхода VFO/TX0/BFO

      quartz(FSK, uint32_t f)          - изменение частоты на выходе TX0 на +f/100 (для «цифры»)
      quartz(sideband, uint32_t f)     - изменение частоты на выходе TX0 на (+-sideband) f

      https://github.com/etherkit/Si5351Arduino
   */

enum quartz_device: uint8_t {
   VFO = 16,
   TX0,
   BFO,
   FSK
};

void quartz(uint8_t device, uint32_t f) {
  uint64_t F64 = frequency * SI5351_FREQ_MULT;
  uint64_t f64 =         f * SI5351_FREQ_MULT;

  switch(device) {
    case  Go: geterodin.init(XTAL_CAPACITOR, f, 0); break;
    case FSK: if (sideband == MODE_USB) F64 += f; 
              else                      F64 -= f;
              geterodin.set_freq(F64, SI5351_CLK1); break;
    case BFO: geterodin.set_freq(f64, SI5351_CLK2); break;
    case TX0: geterodin.set_freq(f64, SI5351_CLK1); break;
    case VFO: geterodin.set_freq(f64, SI5351_CLK0); break;
    
    case MODE_LSB: geterodin.set_freq((F64 - f64), SI5351_CLK1); break;
    case MODE_USB: geterodin.set_freq((F64 + f64), SI5351_CLK1); break;
  }
}

void quartz(uint8_t device, uint8_t onoff) {
  uint32_t side = sideband == MODE_USB ? BFO_USB: BFO_LSB;
  uint32_t Fr   = status(TX) ? frequency: frequency + rit;

  switch (onoff) {
    case  ON:
      switch(device) {
        case BFO: quartz(BFO, side);      break;
        case VFO: quartz(VFO, side + Fr); break;
      }
    case OFF: 
      switch(device) {
        case VFO: geterodin.output_enable(SI5351_CLK0, onoff); break;
        case TX0: geterodin.output_enable(SI5351_CLK1, onoff); break;
        case BFO: geterodin.output_enable(SI5351_CLK2, onoff); break;
      }
    break;
  }
}



#include <TM1637mod.h>
TM1637 tm1637(PIN_CLK, PIN_DIO);

uint8_t chr(char c) { // https://www.mathsisfun.com/binary-decimal-hexadecimal-converter.html
  switch (c) {
    case 'A': return 0b01110111;
    case 'b': return 0b01111100;
    case 'C': return 0b00111001;
    case 'c': return 0b01011000;
    case 'd': return 0b01011110;
    case 'E': return 0b01111001;
    case 'F': return 0b01110001;
    case 'H': return 0b01110110;
    case 'h': return 0b01110100;
    case 'I': return 0b00110000;
    case 'i': return 0b00010000;
    case 'J': return 0b00011110;
    case 'L': return 0b00111000;
    case 'l': return 0b00011000;
    case 'N': return 0b00110111;
    case 'n': return 0b01010100;
    case 'O': return 0b00111111;
    case 'o': return 0b01011100;
    case 'P': return 0b01110011;
    case 'q': return 0b01100111;
    case 'R': return 0b00110001;
    case 'r': return 0b01010000;
    case 'S': return 0b01101101;
    case 't': return 0b01111000;
    case 'U': return 0b00111110;
    case 'u': return 0b00011100;
    case 'Y': return 0b01101110;
    case '-': return 0b01000000;
    default : return (uint8_t)10;
  }
}

void showText(char chr0, char chr1, char chr2, char chr3, uint8_t point = 10) {
  digits[0] = chr(chr0);
  digits[1] = chr(chr1);
  digits[2] = chr(chr2);
  digits[3] = chr(chr3);
  digits[4] = point;
}

void val2digits(int16_t value) {
  value %= 10000;
  digits[0] = value / 1000; value %= 1000;
  digits[1] = value / 100;  value %= 100;
  digits[2] = value / 10;
  digits[3] = value % 10;
}

void showEmpty(void) {
  digits[0] = 10;
  digits[1] = 10;
  digits[2] = 10;
  digits[3] = 10;
  digits[4] = 10;
}

void writeToDisplay(void) {
  const  uint32_t timer = 250UL;
  static uint32_t previous;
  static uint8_t  locmem[5];
  bool nochange = true;

  if (!status(TX) && !status(Led)) showEmpty();

  for (uint8_t i = 0; i < 4; i++) {
    if (digits[i] != locmem[i]) nochange = false;
  }
  if (nochange) return;

  if (millis() - previous > timer) {
    for (uint8_t i = 0; i < 4; i++) {
      locmem[i] = digits[i];
      tm1637.point(i == digits[4] ? true: false);
      tm1637.display(i, digits[i]);
    }
    previous = millis();
  }
}

void writeToDisplayNow(void) {
  uint8_t next = status(Led) ? ON: OFF;
  
  status(Led, ON);
  writeToDisplay();
  status(Led, next);
}

void showUint(uint16_t data, uint8_t point = 10) {
  val2digits((int)data);
  digits[4] = point;
  writeToDisplay();     // 1250
}

void ignitePoint(bool on = true) {
  if (digits[4] > 3) return;
  tm1637.point(on);
  tm1637.display(digits[4], digits[digits[4]]);
}

void blinkingPoint(void) {
  static bool     on = true;
  
  if (digits[4] > 3) return;
  if (softTimer(1, 250UL)) {
    on = !on;
    ignitePoint(on);
  }
}

enum show_option: uint8_t {
   Load = 30,
   Ferr,
   Side,
   Mute,
   Safe,
   Blur,
   None,
   db,
   point,
   XTALC,
   Bright, // 40
};

void show(uint8_t key) {
  switch(key) {
    case Load:
         showText('L','O','A','d');
         writeToDisplayNow();
         delay(500);
    break;
    case Safe:
         showText('S','A','F','E');
         writeToDisplayNow();
         delay(500);
    break;
    case Blur:
         showText('b','l','u','r');
         writeToDisplay();
         delay(500);
    break;
    case None:
         showText('c','h','-','-',1);
         writeToDisplay();
         delay(500);
    break;
    case Tx:
         showText('-','o','n','-');
         writeToDisplayNow();
    break;
    case Rx:
         showText('-','n','o','-');
         writeToDisplayNow();   
    break;
    case Ferr:
         showText('F','E','r','r',0);
         writeToDisplayNow();
    break;
    case Mute:
         showText('-','-','d','b',1);
         writeToDisplayNow();
    break;
    case db:
         val2digits((int)soundLevel);
         digits[0] = digits[2] == 0 ? 10: digits[2];
         digits[1] = digits[3];
         digits[2] = chr('d');
         digits[3] = chr('b');
         digits[4] = 1;
         writeToDisplayNow();   // 10.db
    break;
    case point:
         val2digits((int)setPowerVar);
         digits[0] = chr('P');
         digits[4] = 0;
         writeToDisplay();      // P.255
    break;
    case Side:
         digits[0] = 10;
         digits[1] = sideband == MODE_LSB ? chr('L'): chr('U');
         digits[2] = chr('S');
         digits[3] = chr('b');
         digits[4] = 10;
         writeToDisplayNow();   // LSb or USb
         delay(500);
    break;
    case XTALC:
         val2digits((int)(4 + 2 * XTALcapacitor)); // 1:6pf  2:8pf  3:10pf
         digits[0] = digits[2] == 0 ? 10: digits[2];
         digits[1] = digits[3];
         digits[2] = chr('P');
         digits[3] = chr('F');
         digits[4] = 1;
         writeToDisplay();      // 10.PF
    break;
    case Bright:
         uint8_t old = readByteFromEEPROM(memory.brightness);
         digits[0] = chr('b');
         digits[1] = chr('r');
         digits[2] = old == brightness ? 10: 0;
         digits[3] = brightness;
         digits[4] = 1;
         writeToDisplay();      // br. 4
    break;
  }
}



#include <EEPROM.h>

   /* EEPROM functions

      В ..\hardware\tools\avr\avr\include\avr\eeprom.h
      уже существуют готовые функции того же назначения.
      Но пусть будут собственные.
      
      clearEEPROM();
   */

uint8_t readByteFromEEPROM(uint8_t address) {
  uint8_t data;

  EEPROM.get(address, data);
  return data;
}

uint16_t readUintFromEEPROM(uint8_t address) {
  uint16_t data;

  EEPROM.get(address, data);
  return data;
}

uint32_t readLongFromEEPROM(uint8_t address) {
  uint32_t data;

  EEPROM.get(address, data);
  return data;
}

float readFloatFromEEPROM(uint8_t address) {
  uint8_t i;
  uint8_t data[4];

  for (i = 0; i < 4; i++) data[i] = EEPROM.read(i + address);
  float *y = (float *)&data;
  return y[0];
}

void writeDataToEEPROM(uint8_t address, uint8_t data) {
  EEPROM.update(address, data);
  show(Safe);
}

void writeDataToEEPROM(uint8_t address, uint16_t data) {
  EEPROM.put(address, data);
  show(Safe);
}

void writeDataToEEPROM(uint8_t address, uint32_t data) {
  EEPROM.put(address, data);
  show(Safe);
} 

void writeDataToEEPROM(uint8_t address, float data) {
  uint8_t i;
  uint8_t *x = (uint8_t *)&data;

  for (i = 0; i < 4; i++) EEPROM.write(i + address, x[i]);
  show(Safe);
}



   /* The sketch itself */

   /* Софтовый таймер на три канала для не особо точных применений */

bool softTimer(uint8_t slot, uint32_t ms) {
  static uint32_t mem[]   = {0, 0, 0};
  static uint8_t  first[] = {0, 0, 0};

  if (ms == 0) {
    mem[slot] = 0;
    first[slot] = 0;
  } else if (millis() - mem[slot] > ms || (first[slot] == 0 && ms > millis())) {
    mem[slot] = millis();
    first[slot]++;
    return true;
  }
  return false;
}
   
bool inRange(uint8_t mod, uint32_t F0 = frequency) {
  uint32_t F1 = mod == Tx ? FminTX: FminRX;
  uint32_t F2 = mod == Tx ? FmaxTX: FmaxRX;

  if (F0 < F1) return false;
  if (F0 > F2) return false;
  return true;
}

bool keyPress(void) {
   return (bool)digitalRead(BTN_KEY) ? false: true;
}

   /* Frequency Scanner */

void scanFrequency(void) {
  if (softTimer(2, 3000UL)) {
    switch(getScan()) {
      case   UP: frequency += 1000UL; break;
      case DOWN: frequency -= 1000UL; break;
    }
    if (frequency > FmaxRX) frequency = FminRX;
    if (frequency < FminRX) frequency = FmaxRX;
    quartz(VFO, ON);
  }
}

   /* Cat Interface */

bool workMode(void) {
  return menu[0] == 1 ? true: false;
}

void workModeON(void) {
  menu[0] = 1;
}
   
void catSetPtt(uint8_t onoff) {
  if (!workMode()) {
    show(Rx); 
    return;
  }
  if (!inRange(Tx)) {
    show(Ferr);
    return;
  }
  if (onoff) {
    adc.tx(AudioInput);
    tx();
  } else {
    adc.rx();
    rx();
  }
}

void catSetMode(uint8_t mod) {
  if (status(TX)) return;
  if (workMode()) {
    if (mod == CAT_MODE_LSB) sideband = MODE_LSB;
    if (mod == CAT_MODE_USB) sideband = MODE_USB;
    quartz(VFO, ON);
    quartz(BFO, ON);
    show(Side);
  }
}

void catSetFreq(int32_t f) {
  uint32_t Fr = (uint32_t)f;

  if (status(TX)) return;
  if (workMode()) {
    if (frequency != Fr) {
      frequency = Fr;
      quartz(VFO, ON);
    }
  }
}

bool catGetPtt() {
  return status(TX);
}

uint8_t catGetMode() {
  return sideband == MODE_LSB ? CAT_MODE_LSB: CAT_MODE_USB;
}

long catGetFreq() {
  return (int32_t)frequency;
}

   /* AFP-FSK */

void AFP_FSK(uint32_t f) {
  static uint32_t old;
  uint32_t f16;

  if (f == 0) {
     old = f;
     quartz(TX0, OFF);
  }

  f16 = old >= f ? old - f: f - old;
  if (f16 < 20) return;

  f16 = f * 25;
  f16 >>= 2;
  quartz(FSK, f16);
  if (old == 0) quartz(TX0, ON);
  old = f;
}

   /* XTAL */

void setXTALcapacitorPF(uint8_t c) {
  XTALcapacitor = constrain(c, 1, 3); // 1:6pf  2:8pf  3:10pf
  switch (XTALcapacitor) {
    case 1: XTAL_CAPACITOR = SI5351_CRYSTAL_LOAD_6PF;  break;
    case 2: XTAL_CAPACITOR = SI5351_CRYSTAL_LOAD_8PF;  break;
    case 3: XTAL_CAPACITOR = SI5351_CRYSTAL_LOAD_10PF; break;
  }
}

void setXTALcalibration(uint16_t k = XTALcorrector) {
  XTALcorrector = constrain(k, 0, 5000);
  XTALcorrector = XTALcorrector == k ? XTALcorrector: 1500;
  quartz(Go, XTAL_QUARTZ + XTALcorrector);

  geterodin.drive_strength(SI5351_CLK0, SI5351_DRIVE_6MA);
  geterodin.drive_strength(SI5351_CLK1, SI5351_DRIVE_4MA);
  geterodin.drive_strength(SI5351_CLK2, SI5351_DRIVE_6MA);
}

void go(void) {
  quartz(BFO, ON);
  quartz(TX0, OFF);
  quartz(VFO, ON);
}

   /* PWM enable */

void frequencyPWM(uint16_t f) { // https://alexgyver.ru/lessons/pwm-overclock/
  noInterrupts();
  TCCR2B = 0b00000001;
  TCCR2A = f == 31400 ? 0b00000001: 0b00000011; // D3 Timer2 31400 or 62500 Hz
  interrupts();
}

   /* Power */

void powerSetDAC(uint8_t pwr) {
  analogWrite(PIN_DRIVER, pwr);
}

bool powerSet(uint16_t milliwatt) { // всё в мВт!
  static uint8_t counter[] = {0,0};
  static uint16_t mem_mW;
  static bool mem_hi;
  bool hi;
  uint16_t delta;

  if (counter[0] == 0) {
    mem_mW = 0;
    counter[0]++;

    float tmp = // Чтобы заказанная мощность устанавливалась быстрей.
    96.22037 
    +0.0161      * powf(power_mWt, 1) 
    +1.26817E-4  * powf(power_mWt, 2)
    -1.32117E-7  * powf(power_mWt, 3)
    +3.59971E-11 * powf(power_mWt, 4);

    setPowerVar = constrain((uint8_t)tmp, setPowerVarMin, setPowerVarMax);
    powerSetDAC(setPowerVar);  // W variable
    delay(10);
  }

  if (milliwatt > power_mWt) {
    hi = true;
    delta = milliwatt - power_mWt;
    if (setPowerVar > setPowerVarMin) setPowerVar--;
  } else if (milliwatt < power_mWt) {
    hi = false;
    delta = power_mWt - milliwatt;
    if (setPowerVar < setPowerVarMax) setPowerVar++;
  }  
  if (delta == 0 || counter[0] > 10 || counter[1] > 10) {
    mem_mW = 0;
    counter[0] = 0;
    counter[1] = 0;
    power_mWt = milliwatt;
    return true;
  }
  
  if (mem_mW == milliwatt) {
    counter[1]++;
  } else {
    counter[1] = 0;
  }
  
  if (mem_hi != hi) {
    mem_hi = hi;
    counter[0]++;
  }

  mem_mW = milliwatt;
  powerSetDAC(setPowerVar);
  return false;
}

   /* Mathematics */

float polynomialFitting(float x) {        // points from analogRead() to Watt
  float y = 0.0;

  for (uint8_t i = 0; i < sizeof(PowerPolinom)/sizeof(float); i++) {
    y += PowerPolinom[i] * powf(x, i);
  }
  return y;
}

/* Analog measurements */

float averaged(const uint8_t analogPin) {
  static float    buf[] = {0.0, 0.0, 0.0, 0.0};
         uint8_t  i;
         uint8_t  n     = 0;
         uint8_t  lim   = 10;

  switch(analogPin) {
    case PIN_FORWARD: n = 1;           break;
    case PIN_REFLECT: n = 2;           break;
    case PIN_VOLTAGE: n = 3; lim = 99; break;
  }
  for (i = 0; i < lim; i++) {
    buf[n] += ((float)analogRead(analogPin) - buf[n]) * 0.15;
  }
  return buf[n];
}

float wattForward(void) {
  return polynomialFitting(averaged(PIN_FORWARD));
}

float wattReflect(void) {
  return polynomialFitting(averaged(PIN_REFLECT));
}

uint16_t milliWattPower(void) {
  return (uint16_t)(roundf((wattForward() - wattReflect()) * 1000));
}

uint16_t SWR(void) {
  float f = sqrtf(wattForward()); 
  float r = sqrtf(wattReflect());
  float s = 100.0;

  if (f < 0.05) {
     f = 0.05;
     r = 0.0;
  }
  s *= (float)(f + r);
  s /= (float)(f - r);
  s = roundf(s);
  return (uint16_t)s;
}



   /* Voltmeter */

uint16_t pointsToVoltage(uint16_t points) {
  uint32_t val = points;

  val *= 4000000;
  val /= readLongFromEEPROM(memory.supplyVoltage);
  val = constrain(val, 0, 1600);
  return (uint16_t)val;
}

void voltageToPoints(uint16_t volts, uint16_t points) {
  uint32_t val = points;

  val *= 4000000;
  val /= volts;
  writeDataToEEPROM(memory.supplyVoltage, val);
}



   /* Sound
      functions from GyverPWM 
      https://github.com/AlexGyver/GyverLibs/tree/master/GyverPWM
   */

void defaultSetupD9(void) {
  noInterrupts();
  TCCR1A = 0b10100001;
  TCCR1B = 0b00000011;
  interrupts();
}

void squareSetupD9(void) {
  uint32_t top = (float)CPU_CLOCK_FREQ / (float)TUNE_TONE / 2 - 1;
  
  noInterrupts();
  TCCR1A = 0b01010000;
  TCCR1B = 0b00011001;
  TCCR1C = 0b01000000;
  ICR1H  = highByte(top);
  ICR1L  = lowByte(top);
  interrupts();
}

void toneStart(void) {
  TCCR1A &= ~(1 << COM1A1);
  bitWrite(PORTB, 1, 0);
}

void toneStop(void) {
  TCCR1A |= (1 << COM1A1);
}

void control(uint8_t onoff) {
  switch (onoff) {
    case  ON:
      squareSetupD9();
      toneStart();
    break;
    case OFF:
      toneStop();
      defaultSetupD9();
    break;
  }
}

   /* Beeper */

void morseUnit(uint8_t n) {
  control(ON);
  delay(65 * n);
  control(OFF);
  delay(65);
}

void morseSilence(void) {
  delay(65 * 3);
}

void beeper(char n = 'e') {
  setSoundLevel(Cw);
  morseUnit(1);
  switch(n) {
    case 'i': morseUnit(1); break;
  }
  setSoundLevel(Rx);
}



   /* «Morse» Buttons
      https://github.com/GyverLibs/VirtualButton
   */

#define VB_CLICK 500                        // таймаут накликивания VirtualButton, мс
#define VB_DEB 50                           // дебаунс кнопки VirtualButton, мс

#include <VirtualButton.h>

VButton bc;
VButton vu;
VButton vd;
VButton su;
VButton sd;

char MorseKeySwitchValue(void) {

  vu.poll(!digitalRead(BTN_VALUE_UP));  // ↑
  
       if (vu.hasClicks(1)) return 'E'; // •
  else if (vu.hasClicks(2)) return 'I'; // ••
  else if (vu.hasClicks(3)) return 'S'; // •••
  else if (vu.step(0))      return 'T'; // ▬
  else if (vu.step(1))      return 'A'; // •▬
  else if (vu.step(2))      return 'U'; // ••▬
  else if (vu.step(3))      return 'V'; // •••▬

  vd.poll(!digitalRead(BTN_VALUE_DW));  // ↓

       if (vd.hasClicks(1)) return 'e'; // •
  else if (vd.hasClicks(2)) return 'i'; // ••
  else if (vd.hasClicks(3)) return 's'; // •••
  else if (vd.step(0))      return 't'; // ▬
  else if (vd.step(1))      return 'a'; // •▬
  else if (vd.step(2))      return 'u'; // ••▬
  else if (vd.step(3))      return 'v'; // •••▬

  return '0';
}

char MorseKeySwitchSound(void) {

  su.poll(!digitalRead(BTN_SOUND_UP));  // ↑
  
       if (su.hasClicks(1)) return 'E'; // •
  else if (su.hasClicks(2)) return 'I'; // ••
  else if (su.hasClicks(3)) return 'S'; // •••
  else if (su.step(0))      return 'T'; // ▬
  else if (su.step(1))      return 'A'; // •▬
  else if (su.step(2))      return 'U'; // ••▬

  sd.poll(!digitalRead(BTN_SOUND_DW));  // ↓

       if (sd.hasClicks(1)) return 'e'; // •
  else if (sd.hasClicks(2)) return 'i'; // ••
  else if (sd.hasClicks(3)) return 's'; // •••
  else if (sd.step(0))      return 't'; // ▬
  else if (sd.step(1))      return 'a'; // •▬
  else if (sd.step(2))      return 'u'; // ••▬

  return '0';
}

uint8_t MorseKeySwitchFunctional(void) {

  bc.poll(!digitalRead(BTN_SELECTOR)); // ↑

       if (bc.hasClicks(1)) return 5;  // •
  else if (bc.hasClicks(2)) return 2;  // ••
  else if (bc.hasClicks(3)) return 3;  // •••
  else if (bc.step(0))      return 1;  // ▬ 
  else if (bc.step(1))      return 4;  // •▬
  else if (bc.step(2))      return 6;  // ••▬
  else if (bc.step(3))      return 7;  // •••▬ 

  return 0;
}

   /* Sound Volume Control 
      https://github.com/liman324/PT2257
   */

#include <PT2257.h>
PT2257 volume;

void readSoundLevel3(void) {
  soundLevelRX  = constrain(readByteFromEEPROM(memory.soundLevelRX), 0, 70);
  soundLevelTX  = constrain(readByteFromEEPROM(memory.soundLevelTX), 0, 70);
  soundLevelCW  = constrain(readByteFromEEPROM(memory.soundLevelCW), 0, 40);
}

void writeSoundLevel(void) {
  writeDataToEEPROM(memory.soundLevelTX, soundLevelTX);
  writeDataToEEPROM(memory.soundLevelRX, soundLevelRX);
  writeDataToEEPROM(memory.soundLevelCW, soundLevelCW);
}

uint8_t variant(void) {
  return status(TX) ? (status(Tune) ? Cw: Tx): Rx;
}

uint8_t getSoundLevel(void) {
  switch (variant()) {
    case Rx: return soundLevelRX; break;
    case Cw: return soundLevelCW; break;
    case Tx: return soundLevelTX; break;
  }
}

void saveSoundLevel(uint8_t soundLevel) {
  switch (variant()) {
    case Rx: soundLevelRX = soundLevel; break;
    case Cw: soundLevelCW = soundLevel; break;
    case Tx: soundLevelTX = soundLevel; break;
  }
}

void setSoundLevel(uint8_t sh) {
  switch (sh == Auto ? variant(): sh) {
    case Rx: volume.Right(0); volume.Left (soundLevelRX); break;
    case Cw: volume.Left(0);  volume.Right(soundLevelCW); break;
    case Tx: volume.Left(0);  volume.Right(soundLevelTX); break;
  }
}

void sound(uint16_t delaytime) {
  static uint32_t previous;
  const  uint32_t timer = 500UL;
  bool change = true;

  soundLevel = getSoundLevel();
  switch (MorseKeySwitchSound()) {
    case 't': if (soundLevel >= 10) soundLevel -= 9;
    case 'e': if (soundLevel >=  1) soundLevel -= 1; break;
    case 'T': if (soundLevel <= 69) soundLevel += 9;
    case 'E': if (soundLevel <= 78) soundLevel += 1; break;
    case 'a': soundLevel =  0; break;
    case 'A': soundLevel = 79; break;
    case 'u': volume.Mute(true);  show(Mute); delay(delaytime); previous = millis(); return; break;
    case 'U': volume.Mute(false); break;
    case 'i': readSoundLevel3(); show(Load); break;
    case 'I': writeSoundLevel(); show(Safe); delay(delaytime); previous = millis(); return; break;
    default:  change = false;
  }

  if (change) {
    saveSoundLevel(soundLevel);
    setSoundLevel(Auto);
    show(db);
    delay(delaytime);
    previous = millis();
  }
  
  if (delaytime == 0 && millis() - previous > timer) {
    show(Tx);
    previous = millis();
  }
}

   /* Modes Switcher */

void PTT(uint8_t onoff) {
  switch (onoff) {
    case ON:
      digitalWrite(PIN_RX, LOW);
      delay(100);
      powerSetDAC(setPowerVar);
      status(TX, ON);
    break;
    case OFF:
      powerSetDAC(OFF);
      delay(100);
      digitalWrite(PIN_RX, HIGH);
      status(TX, OFF);
    break;
  }
}

void tx(void) {
  quartz(VFO, OFF);
  quartz(BFO, OFF);
  pciClear();
  PTT(ON);
  setSoundLevel(Auto);
}

void rx(void) {
  quartz(TX0, OFF);
  PTT(OFF);
  pciSetup(BTN_KEY);
  setSoundLevel(Auto);
  quartz(VFO, ON);
  quartz(BFO, ON);
}

void tuner(uint8_t onoff) {
  switch(onoff) {
    case ON:
      status(Tune, ON);
      quartz(sideband, TUNE_TONE);
      tx();
    break;
    case OFF:
      status(Tune, OFF);
      rx();
    break;
  }
}

void manipulationCW(void) {
  if ((menu[5] == 21 && tuningEnd && keyPress()) || !keyPress()) {
    quartz(TX0, OFF);
    if (status(CW)) control(OFF);
  } else {
    if (status(CW)) control(ON);
    quartz(TX0, ON);
  }
}


  /* Channels */

uint8_t channelGetAddres(uint8_t number) {
  return memory.channel + number * 4;
}

void channelDelete(void) {
  const  uint8_t  no   = 255;
         uint8_t  addr = channelGetAddres(channel);

  EEPROM.update(addr,   no);
  EEPROM.update(addr+1, no);
  EEPROM.update(addr+2, no);
  EEPROM.update(addr+3, no);
}

bool channelRead(bool read = true) {
         uint8_t  addr = channelGetAddres(channel);
         uint32_t freq = readLongFromEEPROM(addr);
         uint8_t  side = bitRead(freq, 31);

  freq &= 0b00000000011111111111111111111111;
  
  if (inRange(Rx, freq)) {
    if (read) {
      frequency = freq;
      sideband  = side;
      go();
    }
    return true;
  }
  return false;
}

void channelWrite(void) {
         uint8_t  addr = channelGetAddres(channel);
         uint32_t freq = frequency;

  freq &= 0b00000000011111111111111111111111;  // max 8388607 Hz
  if (sideband == CAT_MODE_USB) bitSet(freq, 31);

  writeDataToEEPROM(addr, freq);
}

void scanChannels(void) {
         uint8_t  i = channel;

  if (softTimer(2, 3000UL)) {
NEXT_CHANNEL:
    switch(getScan()) {
      case DOWN: channel -= 1; break;
      case   UP: channel += 1; break;
    }
    if (channel <  1) channel = 50;
    if (channel > 50) channel = 1;
    if (!channelRead()) {
      if (i == channel) setScan(OFF);
      else goto NEXT_CHANNEL;
    }
  }
}

void channelStatus(void) {
  val2digits((int)channel);
  digits[0] = chr(channelRead(false) ? 'C': 'c');
  digits[1] = chr('h');
  digits[4] = 1;
  writeToDisplay();      // [C-c]h[.]xx
}

void showRIT(void) {
         int16_t  absrit = abs(rit);

  val2digits(absrit);
  digits[0] = rit < 0 ? chr('-'): 10;
  digits[4] = 10;
  writeToDisplay();      // [C-c]h[.]xx
}

  /* Для сокращения кода */

uint16_t changeUint(char button, uint16_t f, uint8_t mem, uint16_t min, uint16_t max) {
                  switch (button) {
    /* ••▬  */      case 'u': if (f >= min + 1000) f -= 900;
    /* •▬   */      case 'a': if (f >= min +  100) f -=  90;
    /* ▬    */      case 't': if (f >= min +   10) f -=   9;
    /* •    */      case 'e': if (f >  min       ) f -=   1; break;
    /* ••▬  */      case 'U': if (f <= max - 1000) f += 900;
    /* •▬   */      case 'A': if (f <= max -  100) f +=  90;
    /* ▬    */      case 'T': if (f <= max -   10) f +=   9;
    /* •    */      case 'E': if (f <  max       ) f +=   1; break;
    /* ••   */      case 'i': if (mem > 0) { f = readUintFromEEPROM(mem); show(Load); } break;
    /* ••   */      case 'I': if (mem > 0) writeDataToEEPROM(mem, (uint16_t)f); break;
                    default : click = false;
                  }
  return f;
}

uint8_t changeByte(char button, uint8_t f, uint8_t mem, uint8_t min, uint8_t max) {
                  switch (button) {
    /* •▬   */      case 'a': if (f >= min +  100) f -=  90;
    /* ▬    */      case 't': if (f >= min +   10) f -=   9;
    /* •    */      case 'e': if (f >  min       ) f -=   1; break;
    /* •▬   */      case 'A': if (f <= max -  100) f +=  90;
    /* ▬    */      case 'T': if (f <= max -   10) f +=   9;
    /* •    */      case 'E': if (f <  max       ) f +=   1; break;
    /* ••   */      case 'i': if (mem > 0) { f = readByteFromEEPROM(mem); show(Load); } break;
    /* ••   */      case 'I': if (mem > 0) writeDataToEEPROM(mem, (uint8_t)f); break;
                    default : click = false;
                  }
  return f;
}



// INI

void setup() {

  analogReference(INTERNAL);
  
  pinMode(BTN_KEY,      INPUT_PULLUP);  
  pinMode(BTN_SELECTOR, INPUT_PULLUP);
  
  pinMode(BTN_VALUE_UP, INPUT_PULLUP);  
  pinMode(BTN_VALUE_DW, INPUT_PULLUP);

  pinMode(BTN_SOUND_UP, INPUT_PULLUP);
  pinMode(BTN_SOUND_DW, INPUT_PULLUP);    // buttons to ground

  pinMode(PIN_RX,       OUTPUT);
  pinMode(PIN_DRIVER,   OUTPUT);
  pinMode(PIN_TONE_D9,  OUTPUT);

  frequencyPWM(31400);                    // D3 (PIN_DRIVER) frequency PWM 31400 or 62500 Hz

  readSoundLevel3();
  volume.Mute(false);                     // true - on mute  | false - off mute

  brightness = constrain(readByteFromEEPROM(memory.brightness), 0, 7);
  tm1637.init();
  tm1637.set(brightness);
  status(Led, ON);

  setPowerVar   = constrain(setPowerVar, setPowerVarMin, setPowerVarMax);
  power_mWt     = constrain(readUintFromEEPROM(memory.power), 0, power_mWt_Max);
  sideband      = constrain(readByteFromEEPROM(memory.sideband), MODE_LSB, MODE_USB);
  frequency     = constrain(readLongFromEEPROM(memory.frequency), FminRX, FmaxRX);
  frequency     = FminRX < frequency && frequency < FmaxRX ? frequency: F0;
  
  setXTALcapacitorPF(constrain(readByteFromEEPROM(memory.XTALcapacitor), 1, 3)); // 1:6pf 2:8pf 3:10pf
  setXTALcalibration(constrain(readUintFromEEPROM(memory.XTALcorrector), 0, 5000));

  interface(Go);
  rx();

}    // setup

void loop() {

  static uint16_t PrevTimer = 0;
         bool     processUI = false;

  switch (statut(0b00000110)) { // Status (Scan1-Scan0-swrmode1-swrmode2-CW-TX-Tune-Led)
    case   4: if ((TCNT1 - PrevTimer) >= (CPU_CLOCK_FREQ / 1000)) {
                PrevTimer = TCNT1;
                processUI = true;
                setScan(OFF);
                interface(Check);
                sound(0);
              }
              processAudioInput(processUI);
              return;
              break;
    case   6: manipulationCW(); break;
  }

  static uint16_t wm;
  static bool     metering  = true;
         char     button    = MorseKeySwitchValue();
         uint8_t  w0        = MorseKeySwitchFunctional();
         uint16_t w1, w2, w3;

  sound(500);
  interface(Check);

  click = true;
              switch (w0) {
    /* •    */  case  5: menu[menu[0]]++; break;
    /* ▬    */  case  1: workModeON(); break;
    /* ••   */  case  2:
    /* •••  */  case  3: if (!inRange(Tx)) { show(Ferr); workModeON(); break; }
    /* •▬   */  case  4:
    /* ••▬  */  case  6:
    /* •••▬ */  case  7: if (workMode()) menu[0] = w0; break;
                default: click = false;
              }

  /* Принудительное выключение чего-то активированного пунктом меню при выходе из него. */

  if (click) {
              switch(menu[5]) {
                case 11:
                case 12: setScan(OFF); ignitePoint(); break;
                case 21:
                case 22:
                case 31: tuner(OFF); status(CW, OFF); break;
                case 61: metering = true; break;
              }
              menu[5] = menu[menu[0]] + menu[0] * 10;
    beeper();
  }
  click = true;

  switch (menu[0]) {
    case  1:                              // Frequency [▬] → Channels ←← 
      switch (menu[1]) {
        case  1:
          switch(getScan()) {
            case OFF:
              switch (button) {
    /* •••  */  case 's': catSetMode(CAT_MODE_LSB); break;
    /* •••  */  case 'S': catSetMode(CAT_MODE_USB); break;
    /* •••▬ */  case 'v': setScan(DOWN); break;
    /* •••▬ */  case 'V': setScan(UP);   break;
    /* ••   */  case 'i': frequency = readLongFromEEPROM(memory.frequency);
                          sideband  = readByteFromEEPROM(memory.sideband);
                          button = 'n';
                          show(Load);
                break;
    /* ••   */  case 'I': writeDataToEEPROM(memory.frequency, (uint32_t)frequency); 
                          writeDataToEEPROM(memory.sideband, (uint8_t)sideband);
                          button = 'n';
                break;
                default : click = false;
              }
              if (click) { go(); beeper(); }
              click = true;
              
              w1 = frequency / 1000;
              w2 = FminRX / 1000;
              w3 = FmaxRX / 1000;
              w1 = changeUint(button, w1, OFF, w2, w3);
                   showUint(w1, 0);

              if (click) catSetFreq((int32_t)w1 * 1000);
            break;
            default:
              w1 = frequency / 1000;
              showUint(w1, 0);
              blinkingPoint();
              switch (button) {
    /* •    */  case 'e':
    /* •    */  case 'E':
    /* ▬    */  case 't':
    /* ▬    */  case 'T': setScan(OFF); ignitePoint(); break;
                default : click = false;
              }
              scanFrequency();
          }
        break;
        case  2:                          // Channels [▬] → [•]
               channelStatus();
          switch(getScan()) {
            case OFF:
              switch (button) {
    /* •▬   */  case 'a': channel =  1; break;
    /* •▬   */  case 'A': channel = 50; break;
    /* •••▬ */  case 'v': setScan(DOWN); break;
    /* •••▬ */  case 'V': setScan(UP);   break;
    /* •••  */  case 's': channelDelete(); show(Blur); break;
    /* •••  */  case 'S': channelWrite(); break;
    /* ••   */  case 'i':
    /* ••   */  case 'I': show(channelRead() ? Load: None); break;
                default : click = false;
              }
              if (click) { button = 'n'; beeper(); }
              click = true;
              channel = changeByte(button, channel, OFF, 1, 50);
            break;
            default:
              blinkingPoint();
              switch (button) {
    /* •    */  case 'e':
    /* •    */  case 'E':
    /* ▬    */  case 't':
    /* ▬    */  case 'T': setScan(OFF); ignitePoint(); break;
                default : click = false;
              }
              scanChannels();
          }
        break;
        default:  menu[1] = 1; return;
      }
    break;
    case  2:                              // [••] Power-meter-Watts-Set ←←

              tuner(ON);
              status(CW, ON);

      switch (menu[2]) {
        case  1:                          //  Power-meter-Watts (Set in Watt) [••]

              if (keyPress() && !tuningEnd) {
                w1 = milliWattPower();
                w1 = constrain(w1, 0, power_mWt_Max);
                showUint(w1, 0);
                if (powerSet(w1)) tuningEnd = true;
                click = false;
              } else {
                power_mWt = changeUint(button, power_mWt, memory.power, 0, power_mWt_Max);
                showUint(power_mWt, 0);
                if (click) tuningEnd = false;
              }
        break;
        case  2:                          // Power-meter-Watts (Set in points) [••] → [•]

              if (keyPress()) {
                w1 = milliWattPower();
                w1 = constrain(w1, 0, power_mWt_Max);
                showUint(w1, 0);
                click = false;
              } else {
                setPowerVar = changeByte(button, setPowerVar, OFF, setPowerVarMin, setPowerVarMax);
                show(point);
              }
        break;
        default:  menu[2] = 1; return;
      }
    break;
    case  3:                              // [•••] SWR-meter 01.00-20.00 ←←

              tuner(ON);
              status(CW, ON);

              if (keyPress()) {
                switch (getSWR()) {
                  case Reflect: w1 = averaged(PIN_REFLECT); break;
                  case Forward: w1 = averaged(PIN_FORWARD); break;
                  default:      w1 = SWR();
                }
              } else w1 = 0;
              
              if (getSWR() == OFF) showUint(constrain(w1, 100, 2000), 1); 
              else                 showUint(constrain(w1,   0, 1023));
              switch (button) {
    /* •    */  case 'e': setSWR(getSWR() == OFF ? Reflect: OFF); break;
    /* •    */  case 'E': setSWR(getSWR() == OFF ? Forward: OFF); break;
                default : click = false;
              }
    break;
    case  4:                              // [•▬] RIT ←←
              switch (button) {
    /* ••   */  case 'i':
    /* ••   */  case 'I': rit = 0; go(); break;
              }
              w1 = 1000;
              w1 += rit;
              w1 = changeUint(button, w1, OFF, 1, 1999);
              rit = (int)w1;
              rit -= 1000;
              
              showRIT();
              if (click) go();
    break;
    case  6:                              // [••▬] Voltmeter → Brightness ←←
      switch (menu[6]) {
        case  1:                          // Voltmeter [••▬]
              w1 = averaged(PIN_VOLTAGE);
              if (metering) wm = pointsToVoltage(w1);
              w2 = wm;
              switch (button) {
    /* •▬   */  case 'a':
    /* ▬    */  case 't':
    /* •    */  case 'e':
    /* •▬   */  case 'A':
    /* ▬    */  case 'T':
    /* •    */  case 'E': metering = false; break;
    /* ••   */  case 'i':
    /* ••   */  case 'I': metering = true; voltageToPoints(w2, w1); break;
              }
              w2 = changeUint(button, w2, OFF, 0, 1600);
              showUint(w2, 1);
              if (click) wm = w2;
        break;
        case  2:                          // Brightness [••▬] → [•]
              show(Bright);
              brightness = changeByte(button, brightness, memory.brightness, 0, 7);
              if (click) tm1637.set(brightness);
        break;
        default:  menu[6] = 1; return;
      }
    break;
    case  7:                              // [•••▬] XTAL frequency calibration → XTAL capacity calibration ←←
      switch (menu[7]) {
        case  1:                          // XTAL frequency calibration [•••▬]

              showUint(XTALcorrector);
              XTALcorrector = changeUint(button, XTALcorrector, memory.XTALcorrector, 0, 5000);
              if (click) {
                setXTALcalibration(XTALcorrector);
                go();
              }
        break;
        case  2:                          // XTAL capacity calibration [•••▬] → [•]

              XTALcapacitor = changeByte(button, XTALcapacitor, memory.XTALcapacitor, 1, 3);
              if (click) {
                setXTALcapacitorPF(XTALcapacitor);
                setXTALcalibration();
                go();
              }
              show(XTALC);
        break;
        default:  menu[7] = 1; return;
      }
    break;
    default: workModeON(); return;
  }
  if (click) beeper();
  
}   // loop

ISR(PCINT2_vect) {
  if (millis() - debouncePCINT2_vect >= 20 && keyPress()) {
    debouncePCINT2_vect = millis();
    if (status(Led)) status(Led, OFF); else status(Led, ON);
  }
}

// *****************  Audio Wave Length Measurement routines - Kazuhisa Terasaki AG6NS  *****************
// Copyed from https://qrpguys.com/wp-content/uploads/2021/06/fsk_multi_vfo_063021.zip

#define PLL_CALCULATION_PRECISION       4
#define MIN_INPUT_AUDIO_FREQ            300                    // minimum input audio frequency limit is  300Hz
#define MAX_INPUT_AUDIO_FREQ            2700                   // maximum input audio frequency limit is 2700Hz
#define UPDATE_VFO_PERIOD               (CPU_CLOCK_FREQ / 250) // update FSK frequency 250 times/sec. (every 4ms)
#define NO_SIGNAL_PERIOD_THRESHOLD      (CPU_CLOCK_FREQ / 20)  // no signal detection threshold (/20 = 50ms)
#define MIN_SAMPLE_COUNT_FOR_AVERAGING  2                      // minimum sample counts for averaging filter (2)

volatile uint8_t  gMeasuredFullWaveCount = 0;
volatile uint16_t gTimer1OverflowCounter = 0;
volatile uint32_t gCurrentTimer1InputCaptureValue = 0;
volatile uint32_t gUpperHalfLenSum = 0;
volatile uint32_t gLowerHalfLenSum = 0;

inline void resetMeasuredValues(void) {        // reset values
  noInterrupts();
  gMeasuredFullWaveCount = 0;
  gUpperHalfLenSum       = 0;
  gLowerHalfLenSum       = 0;
  interrupts();
}

inline void readAndResetMeasuredValues(
  uint32_t *currentInputCaptureValue,
  uint8_t  *fullWaveCount,
  uint32_t *upperHalfWaveLenSum,
  uint32_t *lowerHalfWaveLenSum) {

  noInterrupts();
  *currentInputCaptureValue = gCurrentTimer1InputCaptureValue;
  *fullWaveCount         = gMeasuredFullWaveCount;
  *upperHalfWaveLenSum   = gUpperHalfLenSum;
  *lowerHalfWaveLenSum   = gLowerHalfLenSum;

  gMeasuredFullWaveCount = 0;
  gUpperHalfLenSum       = 0;
  gLowerHalfLenSum       = 0;
  interrupts();
}

inline uint32_t readCurrentTimer1Value(void) {

  noInterrupts();
  uint16_t counterValue = TCNT1;
  uint32_t currentTimer1Value = ((uint32_t)gTimer1OverflowCounter << 16) | counterValue;

  if ((TIFR1 & (1 << TOV1)) && (counterValue & 0x8000) == 0x0000) {
    // timer1 overflow happened and hasn't handled it yet
    currentTimer1Value += 0x10000;
  }
  interrupts();
  return currentTimer1Value;
}

void processAudioInput(bool checkNoSignal) {
  static bool sIsTransmitting = false;

  // read the length of the last measured audio wave
  uint32_t currentInputCaptureValue;
  uint8_t  inputCaptureEvents;
  uint32_t upperWaveLenSum;
  uint32_t lowerWaveLenSum;
  readAndResetMeasuredValues(&currentInputCaptureValue, &inputCaptureEvents, &upperWaveLenSum, &lowerWaveLenSum);

  static uint32_t sLastVFOUpdatedInputCaptureValue  = 0;
  static uint16_t sCapturedWaveCount                = 0;
  static uint32_t sUpperWaveLenTotal                = 0;
  static uint32_t sLowerWaveLenTotal                = 0;
  static uint32_t sLastValidSignalInputCaptureValue = 0;

  if (inputCaptureEvents > 0) {
    sCapturedWaveCount += inputCaptureEvents;
    sUpperWaveLenTotal += upperWaveLenSum;
    sLowerWaveLenTotal += lowerWaveLenSum;

    if (sLastVFOUpdatedInputCaptureValue == 0) {
      sLastVFOUpdatedInputCaptureValue = currentInputCaptureValue;
    }

    uint32_t totalWaveLength = currentInputCaptureValue - sLastVFOUpdatedInputCaptureValue;
    if (totalWaveLength >= UPDATE_VFO_PERIOD && sCapturedWaveCount >= MIN_SAMPLE_COUNT_FOR_AVERAGING) {

      // measured audio wave length
      uint32_t averageWaveLength =
        ((sUpperWaveLenTotal << PLL_CALCULATION_PRECISION) + (sCapturedWaveCount / 2)) / sCapturedWaveCount +
        ((sLowerWaveLenTotal << PLL_CALCULATION_PRECISION) + (sCapturedWaveCount / 2)) / sCapturedWaveCount;
      // measured audio frequency
      uint32_t audioFreq = (CPU_CLOCK_FREQ << (PLL_CALCULATION_PRECISION * 2)) / averageWaveLength;
      // frequency is in 28.4 fixed point number, 0.0625Hz resolution

      if (((uint32_t)MIN_INPUT_AUDIO_FREQ << PLL_CALCULATION_PRECISION) <= audioFreq
          && audioFreq <= ((uint32_t)MAX_INPUT_AUDIO_FREQ << PLL_CALCULATION_PRECISION)
          && sLowerWaveLenTotal < sUpperWaveLenTotal
          && sUpperWaveLenTotal < (sLowerWaveLenTotal << 1)) {
        // sLowerWaveLenTotal < sUpperWaveLenTotal < sLowerWaveLenTotal * 2

        // found audio signal
        sLastValidSignalInputCaptureValue = currentInputCaptureValue;

        if (sIsTransmitting) {
          AFP_FSK(audioFreq);
        }

        // set this flag at here so we can ignore the first detected frequency which might include some error
        sIsTransmitting = true;
      }

      sLastVFOUpdatedInputCaptureValue = currentInputCaptureValue;
      sCapturedWaveCount = 0;
      sUpperWaveLenTotal = 0;
      sLowerWaveLenTotal = 0;
    }
  }

  if (checkNoSignal && sIsTransmitting) {
    uint32_t currentTimer1Value = readCurrentTimer1Value();
    uint32_t noSignalPeriod = currentTimer1Value - sLastValidSignalInputCaptureValue;
    if (noSignalPeriod > NO_SIGNAL_PERIOD_THRESHOLD) {

      // detected no signal period
      sLastVFOUpdatedInputCaptureValue = 0;
      sCapturedWaveCount = 0;
      sUpperWaveLenTotal = 0;
      sLowerWaveLenTotal = 0;

      resetMeasuredValues();
      AFP_FSK(0);
      sIsTransmitting = false;
    }
  }
}

inline uint32_t readTimer1InputCaptureValue(void) {
  uint16_t counterValue = ICR1;
  uint32_t currentTimer1Value = ((uint32_t)gTimer1OverflowCounter << 16) | counterValue;

  if ((TIFR1 & (1 << TOV1)) && (counterValue & 0x8000) == 0x0000) {
    currentTimer1Value += 0x10000;             // timer1 overflow happened and hasn't handled it yet
  }
  return currentTimer1Value;
}

ISR(TIMER1_CAPT_vect) {                        // ISR priority 11
  static uint32_t sPrevInputCaptureValue;
  static uint32_t sUpperWaveLen;

  uint32_t currentInputCaptureValue = readTimer1InputCaptureValue();
  uint32_t halfWaveLen = currentInputCaptureValue - sPrevInputCaptureValue;
  uint8_t  currTCCR1B = TCCR1B;

  if (currTCCR1B & (1 << ICES1)) {             // detected Falling Audio Signal Edge (Rising Input Capture Edge)
    static uint32_t sAveUpperHalfWaveLen = 0;
    sAveUpperHalfWaveLen = (sAveUpperHalfWaveLen + sAveUpperHalfWaveLen + sAveUpperHalfWaveLen + halfWaveLen) >> 2;
    if (halfWaveLen < ((sAveUpperHalfWaveLen >> 2) + (sAveUpperHalfWaveLen >> 4))) { // ignore ripple
      return;
    }
    sUpperWaveLen = halfWaveLen;
  } else {                                     // detected Rising Audio Signal Edge (Falling Input Capture Edge)
    static uint32_t sAveLowerHalfWaveLen = 0;
    sAveLowerHalfWaveLen = (sAveLowerHalfWaveLen + sAveLowerHalfWaveLen + sAveLowerHalfWaveLen + halfWaveLen) >> 2;
    if (halfWaveLen < ((sAveLowerHalfWaveLen >> 2) + (sAveLowerHalfWaveLen >> 4))) { // ignore ripple
      return;
    }
    gUpperHalfLenSum += sUpperWaveLen;
    gLowerHalfLenSum += halfWaveLen;
    gCurrentTimer1InputCaptureValue = currentInputCaptureValue;
    gMeasuredFullWaveCount++;
  }
  sPrevInputCaptureValue = currentInputCaptureValue;

  TCCR1B = currTCCR1B ^ (1 << ICES1);          // flip edge selection
  TIFR1 = (1 << ICF1);                         // clear Input Capture interrupt Flag 
                                               // (flipping the edge selection causes the unwanted interrupt)
}

ISR(TIMER1_OVF_vect) {                         // ISR priority 14
  gTimer1OverflowCounter++;
}

// ********************************** End of Kazuhisa Terasaki routines *********************************
